//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2005-2010 Xilinx, Inc.
// This design is confidential and proprietary of Xilinx, Inc.
// All Rights Reserved.
///////////////////////////////////////////////////////////////////////////////
//   ____  ____
//  /   /\/   /
// /___/  \  /   Vendor: Xilinx
// \   \   \/    Version: 12.1
//  \   \        Application: Partial Reconfiguration
//  /   /        Filename: dvi.v
// /___/   /\    Date Last Modified: 11 July 2009
// \   \  /  \
//  \___\/\___\
// Device: Virtex-6
// Design Name: Color2
// Purpose: Partial Reconfiguration User Guide
///////////////////////////////////////////////////////////////////////////////

module dvi (rst_n,
	    pixel_clock,
	    DVI_R,
	    DVI_G,
	    DVI_B,
	    DVI_HSYNC,
	    DVI_VSYNC,
	    DATA_ENABLE
	    );

   input rst_n; // Active high reset, synchronous with 27MHz clock
   input pixel_clock;
   output [7:0] DVI_R, DVI_G, DVI_B; // Outputs to DAC
   output DVI_HSYNC, DVI_VSYNC; // Sync outputs to DVI connector
   output DATA_ENABLE; // DE signal used to indicate when we are not blanking horizontally or vertically

   ////////////////////////////////////////////////////////////////////////////
   //
   // Timing values
   //
   ////////////////////////////////////////////////////////////////////////////

   // 1024 X 768 @ 75Hz with a 78.750MHz pixel clock
`define H_ACTIVE	1024 // pixels
`define H_FRONT_PORCH	  16 // pixels
`define H_SYNCH		  96 // pixels
`define H_BACK_PORCH	 176 // pixels
`define H_TOTAL		1312 // pixels

`define V_ACTIVE	 768 // lines
`define V_FRONT_PORCH	   1 // lines
`define V_SYNCH		   3 // lines
`define V_BACK_PORCH	  28 // lines
`define V_TOTAL		 800 // lines

   ////////////////////////////////////////////////////////////////////////////
   //
   // Internal signals
   //
   ////////////////////////////////////////////////////////////////////////////

   wire pixel_clock;
   reg prst, pixel_reset; // Active high reset, synchronous with pixel clock

   reg [7:0] DVI_R, DVI_B, DVI_G;
   wire dvi_out_blank_b;
   reg hsync1, hsync2, DVI_HSYNC, vsync1, vsync2, DVI_VSYNC;
   // DE
   wire DATA_ENABLE;

   reg [10:0] pixel_count; // Counts pixels in each line
   reg [10:0] line_count; // Counts lines in each frame




   assign reset = ~rst_n;

   always @(posedge pixel_clock)
     begin
	prst <= reset;
	pixel_reset <= prst;
     end

   ////////////////////////////////////////////////////////////////////////////
   //
   // Pixel and Line Counters
   //
   ////////////////////////////////////////////////////////////////////////////

   always @(posedge pixel_clock)
     if (pixel_reset)
       begin
	  pixel_count <= 0;
	  line_count <= 0;
       end
     else if (pixel_count == (`H_TOTAL-1)) // last pixel in the line
       begin
	  pixel_count <= 0;
	  if (line_count == (`V_TOTAL-1)) // last line of the frame
	    line_count <= 0;
	  else
	    line_count <= line_count + 1;
       end
     else
       pixel_count <= pixel_count +1;

   ////////////////////////////////////////////////////////////////////////////
   //
   // Sync and Blank Signals
   //
   ////////////////////////////////////////////////////////////////////////////

   always @ (posedge pixel_clock)
     begin
	if (pixel_reset)
	  begin
	     hsync1 <= 1;
	     hsync2 <= 1;
	     DVI_HSYNC <= 1;
	     vsync1 <= 1;
	     vsync2 <= 1;
	     DVI_VSYNC <= 1;
	  end
	else
	  begin

	     // Horizontal sync
	     if (pixel_count == (`H_ACTIVE+`H_FRONT_PORCH))
	       hsync1 <= 0; // start of h_sync
	     else if (pixel_count == (`H_ACTIVE+`H_FRONT_PORCH+`H_SYNCH))
	       hsync1 <= 1; // end of h_sync

	     // Vertical sync
	     if (pixel_count == (`H_TOTAL-1))
	       begin
		  if (line_count == (`V_ACTIVE+`V_FRONT_PORCH))
		    vsync1 <= 0; // start of v_sync
		  else if (line_count == (`V_ACTIVE+`V_FRONT_PORCH+`V_SYNCH))
		    vsync1 <= 1; // end of v_sync
	       end

	  end

	// Delay hsync and vsync by two cycles to compensate for 2 cycles of
	// pipeline delay in the DAC.
	hsync2 <= hsync1;
	DVI_HSYNC <= hsync2;
	vsync2 <= vsync1;
	DVI_VSYNC <= vsync2;

     end

   // Blanking
   assign dvi_out_blank_b = ((pixel_count<`H_ACTIVE) & (line_count<`V_ACTIVE));

   // DE GENERATION
   assign DATA_ENABLE = dvi_out_blank_b;

   // Composite sync
//   assign dvi_out_sync_b = hsync1 ^ vsync1;

   ////////////////////////////////////////////////////////////////////////////
   //
   // Generate a pretty picture
   //
   ////////////////////////////////////////////////////////////////////////////

   reg [15:0] frame_count;

   always @(posedge pixel_clock)
     if (pixel_reset)
       frame_count <= 0;
     else if ((pixel_count == `H_TOTAL-1) && (line_count == `V_TOTAL-1))
       frame_count <= frame_count + 1;

   wire [6:0] grad;
   assign grad = pixel_count[5:0]+line_count[5:0];



// ColorBar
   always @ (posedge pixel_clock)
      begin if (~dvi_out_blank_b)
	 begin
	    DVI_R <= 0;
	    DVI_G <= 0;
	    DVI_B <= 0;
	    end
	 else

	    begin
	       // ColorBar
	       DVI_R <= ((pixel_count[8] & 1'b1)  ? 8'hff : 0);
	       DVI_G <= ((pixel_count[7] & 1'b1)  ? 8'hff : 0);
	       DVI_B <= ((pixel_count[6] & 1'b1)  ? 8'hff : 0);

	       // Gradient
	       // DVI_R <= (line_count[7]^pixel_count[7]) ? grad*2 : 255-grad*2;
	       // DVI_G <= (line_count[8]^pixel_count[8]) ? grad*2 : 0;
	       // DVI_B <= (line_count[9]^pixel_count[9]) ? grad*2 : 0;

	       // White
	       //DVI_R <= (8'hff);
	       //DVI_G <= (8'hff);
	       //DVI_B <= (8'hff);

	    end

          end
endmodule
