--<insert: c:\HW\releasedULD\headers\DAC_SPI.head>
-- -----------------------------------------------------------------------------
--
-- module:    DAC_SPI
-- project:   wave_gen
-- company:   Xilinx, Inc.
-- author:    WK, AW
-- 
-- comment:
--   This module takes each sample generated by the sample generator  and
--   sends it to the digital to analog converter (DAC) using the  SPI
--   protocol. Each SPI cycle is 32 clocks long, consisting of an  address,
--   command, and 16 bits of data (MSbit first). The DAC on the  development
--   boards use only the most significant 12 out of the 16 bit samples,  but
--   all 16 bits are transmitted.
-- 
-- known issues:
-- status           id     found     description                      by fixed date  by    comment
-- 
-- version history:
--   version    date    author     description
--    11.1-001 20 APR 2009 WK       New for version 11.1            
-- 
-- ---------------------------------------------------------------------------
-- 
-- disclaimer:
--   Disclaimer: LIMITED WARRANTY AND DISCLAMER. These designs  are
--   provided to you as is . Xilinx and its licensors make, and  you
--   receive no warranties or conditions, express,  implied,
--   statutory or otherwise, and Xilinx specifically disclaims  any
--   implied warranties of merchantability, non-infringement,  or
--   fitness for a particular purpose. Xilinx does not warrant  that
--   the functions contained in these designs will meet  your
--   requirements, or that the operation of these designs will  be
--   uninterrupted or error free, or that defects in the  Designs
--   will be corrected. Furthermore, Xilinx does not warrant  or
--   make any representations regarding use or the results of  the
--   use of the designs in terms of correctness,  accuracy,
--   reliability, or  otherwise.
--   
-- LIMITATION OF LIABILITY. In no event will Xilinx or  its
--   licensors be liable for any loss of data, lost profits,  cost
--   or procurement of substitute goods or services, or for  any
--   special, incidental, consequential, or indirect  damages
--   arising from the use or operation of the designs  or
--   accompanying documentation, however caused and on any  theory
--   of liability. This limitation will apply even if  Xilinx
--   has been advised of the possibility of such damage.  This
--   limitation shall apply not-withstanding the failure of  the
--   essential purpose of any limited remedies  herein.
--   
-- Copyright  2002, 2008, 2009 Xilinx,  Inc.
--   All rights reserved
-- 
-- -----------------------------------------------------------------------------
--

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

library work;
use work.wave_gen_pkg.all;


entity DAC_SPI is
    Port (clk_tx         : in  STD_LOGIC;                    -- transmitter clock
          rst_clk_tx     : in  STD_LOGIC;                    -- reset signal synchronized to clk_tx
          en_clk_samp    : in  STD_LOGIC;                    -- indication that the next rising edge of clk_tx will coincide with the rising edge of clk_smap
          samp           : in  STD_LOGIC_VECTOR(15 downto 0);-- the current sample being output. Only valid when samp_val is asserted
          samp_val       : in  STD_LOGIC;                    -- a vaild sample is being output. Asserted for one clk_samp period for each sample
          spi_clk_o      : out STD_LOGIC;                    -- SPI clock
          SPI_MOSI_o     : out STD_LOGIC;                    -- SPI master-out-slave-in datum
          DAC_cs_n_o     : out STD_LOGIC;                    -- DAC SPI chip select (active low)
          DAC_clr_n_o    : out STD_LOGIC                     -- DAC clear (active low)
          );
end DAC_SPI;


architecture Behavioral of DAC_SPI is

       signal data_to_send     : std_logic_vector(31 downto 0) := (others=>'U');
       signal start_spi        : std_logic := 'U';
       signal active           : std_logic := 'U';
       signal old_active       : std_logic := 'U';
       signal stretched_active : std_logic := 'U';

    begin
    
       -- Generate dac_clr_n - asserted only when internal reset is asserted
       DACrst: process (clk_tx)
          begin
             if rising_edge(clk_tx) then                  -- process occurs synchonously with the tx clock
                if (rst_clk_tx = '1') then                -- has the internal reset been asserted?
                   dac_clr_n_o <= '0';                    -- active low assertion of reset
                else                                      -- do non-reset activities
                   dac_clr_n_o <= '1';                    -- no reset applied
                end if;                                   -- end of reset/non-reset activities
             end if;                                      -- end of synchronous events
          end process DACrst;
          
       -- capture the sample to send  [Write to an Update (Power Up) ch, N + channel A]
        data_to_send <= X"0030" &  samp;                  -- The sample to send
       
       -- when the new sample is captured, start the spi transfer on the next clk_samp
       start_spi <= samp_val and en_clk_samp;
          
       -- generate the serialized outputs
       genSerOut: process (clk_tx)
             variable bit_cnt     : integer range 0 to 32 := 0;
          begin
             if rising_edge(clk_tx) then                  -- this process runs synchronously to the tx clock
                if (rst_clk_tx = '1') then                -- is there a reset in progress?
                   active       <= '0';
                   old_active   <= '0';
                   bit_cnt      := 0;
                   dac_cs_n_o   <= '1';                   -- keep DAC unselected (active low signal)
                   spi_mosi_o   <= '1';                   -- line idle high                
                else                                      -- no reset, do "normal" events
                   old_active <= active;                  -- hold the last value for an extra clock
                   if (active = '0') then                 -- if we're currently not active then
                      dac_cs_n_o  <= '1';                 -- DAC normally disabled
                      spi_mosi_o  <= '1';                 -- line normally idles high
                      if (start_spi = '1') then           -- has there been a request to start an SPI transfer?
                         active <= '1';                   -- set flag indicating that we're actively in SPI transfer
                         dac_cs_n_o <= '0';               -- select the DAC (active low)
                         spi_mosi_o <= data_to_send(31);  -- send the first bit of the sequence
                         bit_cnt    := 30;                -- 32 bits to send, but we just sent one
                      end if;                             -- end of start SPI request test
                   else                                   -- we are active
                      spi_mosi_o <= data_to_send(bit_cnt);-- send the next bit
                      if (bit_cnt = 0) then               -- did we just send the last bit?
                         active <= '0';                   -- indicate to the other processes that we're done sending
                      else                                -- still in the process of sending
                         bit_cnt := bit_cnt - 1;          -- count the bit that was just sent
                      end if;                             -- end of bit count check
                   end if;                                -- end of active tests
                end if;                                   -- end reset test 
             end if;                                      -- end synchronous events
          end process genSerOut;     
    
      -- Generate the SPI clock using the DDR flop
      stretched_active <= active or old_active;
      out_ddr_flop_spi_clk_i0: out_ddr_flop  
          port map(clk    => clk_tx,
                   rst    => rst_clk_tx,
                   d_rise => vcc(0),
                   d_fall => stretched_active,
                   q         => spi_clk_o);
                   

    end Behavioral;

