--<insert: c:\HW\releasedULD\headers\debouncer.head>
-- -----------------------------------------------------------------------------
--
-- module:    debouncer
-- project:   wave_gen
-- company:   Xilinx, Inc.
-- author:    WK, AW
-- 
-- comment:
--   Simple switch debouncer. Filters out any transition that lasts less than
--   FILTER clocks long
-- 
-- known issues:
-- status           id     found     description                      by fixed date  by    comment
-- 
-- version history:
--   version    date    author     description
--    11.1-001 20 APR 2009 WK       First version for 11.1          
-- 
-- ---------------------------------------------------------------------------
-- 
-- disclaimer:
--   Disclaimer: LIMITED WARRANTY AND DISCLAMER. These designs  are
--   provided to you as is . Xilinx and its licensors make, and  you
--   receive no warranties or conditions, express,  implied,
--   statutory or otherwise, and Xilinx specifically disclaims  any
--   implied warranties of merchantability, non-infringement,  or
--   fitness for a particular purpose. Xilinx does not warrant  that
--   the functions contained in these designs will meet  your
--   requirements, or that the operation of these designs will  be
--   uninterrupted or error free, or that defects in the  Designs
--   will be corrected. Furthermore, Xilinx does not warrant  or
--   make any representations regarding use or the results of  the
--   use of the designs in terms of correctness,  accuracy,
--   reliability, or  otherwise.
--   
-- LIMITATION OF LIABILITY. In no event will Xilinx or  its
--   licensors be liable for any loss of data, lost profits,  cost
--   or procurement of substitute goods or services, or for  any
--   special, incidental, consequential, or indirect  damages
--   arising from the use or operation of the designs  or
--   accompanying documentation, however caused and on any  theory
--   of liability. This limitation will apply even if  Xilinx
--   has been advised of the possibility of such damage.  This
--   limitation shall apply not-withstanding the failure of  the
--   essential purpose of any limited remedies  herein.
--   
-- Copyright  2002, 2008, 2009 Xilinx,  Inc.
--   All rights reserved
-- 
-- -----------------------------------------------------------------------------
--

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;
use IEEE.MATH_REAL.ALL;

library WORK;
use work.wave_gen_pkg.ALL;                             -- load components


entity debouncer is
    generic ( FILTER     : integer := 200_000_000      -- number of clocks required to acknowledge a valid change in switch state
           );
   port (clk             : in  STD_LOGIC;              -- clock input
         rst             : in  STD_LOGIC;              -- active high reset - synchronous to clock
         signal_in       : in  STD_LOGIC;              -- un-debounced signal
         signal_out      : out STD_LOGIC               -- debounced signal
         );
    constant RELOAD      : integer := FILTER - 1;
end debouncer;


architecture Behavioral of debouncer is
       
       signal signal_in_clk : std_logic := 'U';
       
    begin

       -- Anti-meta-stability/synchronization circuit
       meta_harden_signal_in_i0: meta_harden
                   port map (clk_dst    => clk,
                            rst_dst        => rst,
                            signal_src     => signal_in,
                            signal_dst     => signal_in_clk);

       -- count FILTER number of clocks from a transition to see if the transition has held
       dbnc: process (clk)
             variable signal_out_reg    : std_logic := 'U';
             variable count             : integer range 0 to FILTER := FILTER;
          begin
             if rising_edge(clk) then                     -- everything happens synchronously with the driving clock
                if (rst = '1') then                       -- if the reset is asserted
                   count      := 0;                       -- start with an expired counter
                   signal_out <= signal_in_clk;           -- during reset, the signal is not debounced
                   signal_out_reg := signal_in_clk;       -- initializes to the last value of signal_in when exiting reset
                else                                      -- reset not asserted - do "normal" activities                 
                   if (signal_in /= signal_out_reg) then  -- if the current input is different than the last debounced output...
                      if (count = 0) then                 -- if the count has expired
                         signal_out <= signal_in_clk;     -- take on the new value
                         count := RELOAD;                 -- reload the counter
                      else                                -- count not done
                         count := count - 1;              -- decrement the counter
                      end if;                             -- end of count done test
                   else                                   -- if the signal_in is the same as the previous debounced output, 
                      count := RELOAD;                    -- reload the counter to be ready for the possibility of a transition on the next clock
                   end if;                                -- end of transition check
                   signal_out_reg := signal_in_clk;       -- update the register
                   signal_out <= signal_out_reg;          -- drive the output with the previously latched value                
                end if;                                   -- end of reset/normal operation                
             end if;                                      -- end of synchronous events test      
          end process dbnc;

    end Behavioral;


