`timescale 1ns/1ps

module test_parameters;
  integer StartAddress = 0;
  integer EndAddress = 4096;
  integer DataPattern = 0;
endmodule

module mig_hw_tb;

  // ========================================================================== //
  // Parameters                                                                 //
  // ========================================================================== //

  parameter C3_MEMCLK_PERIOD           = 3000;
  parameter C3_RST_ACT_LOW             = 0;
  parameter C3_INPUT_CLK_TYPE          = "DIFFERENTIAL";
  parameter C3_NUM_DQ_PINS             = 16;
  parameter C3_MEM_ADDR_WIDTH          = 13;
  parameter C3_MEM_BANKADDR_WIDTH      = 3;
  parameter C3_MEM_ADDR_ORDER          = "ROW_BANK_COLUMN";
  parameter C3_P0_BYTE_ADDR_WIDTH      = 30;
  parameter C3_P0_MASK_SIZE            = 16;
  parameter C3_P0_DATA_PORT_SIZE       = 128;
  parameter C3_MEM_BURST_LEN           = 8;
  parameter C3_MEM_NUM_COL_BITS        = 10;
  parameter C3_MC_CALIB_BYPASS         = "NO";
  parameter C3_CALIB_SOFT_IP           = "TRUE";

  // ========================================================================== //
  // Signal Declarations                                                        //
  // ========================================================================== //

  wire [C3_MEM_ADDR_WIDTH-1:0]      mcb3_dram_a;
  wire [C3_MEM_BANKADDR_WIDTH-1:0]  mcb3_dram_ba;
  wire                              mcb3_dram_ck;
  wire                              mcb3_dram_ck_n;
  wire [C3_NUM_DQ_PINS-1:0]         mcb3_dram_dq;
  wire                              mcb3_dram_dqs;
  wire                              mcb3_dram_dqs_n;
  wire                              mcb3_dram_dm;
  wire                              mcb3_dram_ras_n;
  wire                              mcb3_dram_cas_n;
  wire                              mcb3_dram_we_n;
  wire                              mcb3_dram_cke;
  wire                              mcb3_dram_odt;
  wire                              mcb3_dram_udqs;    // for X16 parts
  wire                              mcb3_dram_udqs_n;  // for X16 parts
  wire                              mcb3_dram_udm;     // for X16 parts

  wire                              c3_sys_clk_p;
  wire                              c3_sys_clk_n;
  reg                               c3_sys_rst;
  wire                              c3_sys_rst_n;
  wire                              c3_calib_done;
  reg                               c3_rst_done;

  reg                               c3_clk0;
  wire                              c3_rst0;

  reg                               c3_p0_cmd_en;
  reg  [2:0]                        c3_p0_cmd_instr;
  reg  [5:0]                        c3_p0_cmd_bl;
  reg  [C3_P0_BYTE_ADDR_WIDTH-1:0]  c3_p0_cmd_byte_addr;
  wire                              c3_p0_cmd_empty;
  wire                              c3_p0_cmd_full;
  reg                               c3_p0_wr_en;
  reg  [C3_P0_MASK_SIZE-1:0]        c3_p0_wr_mask;
  reg  [C3_P0_DATA_PORT_SIZE-1:0]   c3_p0_wr_data;
  wire                              c3_p0_wr_full;
  wire                              c3_p0_wr_empty;
  wire [6:0]                        c3_p0_wr_count;
  wire                              c3_p0_wr_underrun;
  wire                              c3_p0_wr_error;
  reg                               c3_p0_rd_en;
  wire [C3_P0_DATA_PORT_SIZE-1:0]   c3_p0_rd_data;
  wire                              c3_p0_rd_full;
  wire                              c3_p0_rd_empty;
  wire [6:0]                        c3_p0_rd_count;
  wire                              c3_p0_rd_overflow;
  wire                              c3_p0_rd_error;

  wire                              rzq3;
  wire                              zio3;

  // ========================================================================== //
  // Clocks Generation                                                          //
  // ========================================================================== //

  initial
    c3_clk0 = 1'b0;
  always
    #5 c3_clk0 = ~c3_clk0;

  // ========================================================================== //
  // Reset Generation                                                           //
  // ========================================================================== //

  initial begin
    c3_rst_done = 0;
    c3_sys_rst = 1'b0;
    #20;
    c3_sys_rst = 1'b1;
    c3_rst_done = 1;
  end

  assign c3_sys_rst_n = C3_RST_ACT_LOW ? c3_sys_rst : ~c3_sys_rst;

  // ========================================================================== //
  // Error Grouping                                                             //
  // ========================================================================== //

  PULLDOWN zio_pulldown3 (.O(zio3));
  PULLDOWN rzq_pulldown3 (.O(rzq3));

  // ========================================================================== //
  // Design Top Instantiation                                                   //
  // ========================================================================== //

  mig_dut #(
    .C3_P0_MASK_SIZE       (C3_P0_MASK_SIZE),
    .C3_P0_DATA_PORT_SIZE  (C3_P0_DATA_PORT_SIZE),
    .C3_MEMCLK_PERIOD      (C3_MEMCLK_PERIOD),
    .C3_RST_ACT_LOW        (C3_RST_ACT_LOW),
    .C3_INPUT_CLK_TYPE     (C3_INPUT_CLK_TYPE),

    .C3_MEM_ADDR_ORDER     (C3_MEM_ADDR_ORDER),
    .C3_NUM_DQ_PINS        (C3_NUM_DQ_PINS),
    .C3_MEM_ADDR_WIDTH     (C3_MEM_ADDR_WIDTH),
    .C3_MEM_BANKADDR_WIDTH (C3_MEM_BANKADDR_WIDTH),
    .C3_MC_CALIB_BYPASS    (C3_MC_CALIB_BYPASS),

    .C3_CALIB_SOFT_IP      (C3_CALIB_SOFT_IP)
  )
  mig_inst (
    .c3_sys_clk_p          (c3_sys_clk_p),
    .c3_sys_clk_n          (c3_sys_clk_n),
    .c3_sys_rst_n          (c3_sys_rst_n),

    .mcb3_dram_dq          (mcb3_dram_dq),
    .mcb3_dram_a           (mcb3_dram_a),
    .mcb3_dram_ba          (mcb3_dram_ba),
    .mcb3_dram_ras_n       (mcb3_dram_ras_n),
    .mcb3_dram_cas_n       (mcb3_dram_cas_n),
    .mcb3_dram_we_n        (mcb3_dram_we_n),
    .mcb3_dram_odt         (mcb3_dram_odt),
    .mcb3_dram_cke         (mcb3_dram_cke),
    .mcb3_dram_ck          (mcb3_dram_ck),
    .mcb3_dram_ck_n        (mcb3_dram_ck_n),
    .mcb3_dram_dqs         (mcb3_dram_dqs),
    .mcb3_dram_dqs_n       (mcb3_dram_dqs_n),
    .mcb3_dram_udqs        (mcb3_dram_udqs),    // for X16 parts
    .mcb3_dram_udqs_n      (mcb3_dram_udqs_n),  // for X16 parts
    .mcb3_dram_udm         (mcb3_dram_udm),     // for X16 parts
    .mcb3_dram_dm          (mcb3_dram_dm),

    .c3_clk0               (c3_clk0),
    .c3_rst0               (c3_rst0),
    .c3_calib_done         (c3_calib_done),
    .mcb3_rzq              (rzq3),
    .mcb3_zio              (zio3),

    .c3_p0_cmd_en          (c3_p0_cmd_en),
    .c3_p0_cmd_instr       (c3_p0_cmd_instr),
    .c3_p0_cmd_bl          (c3_p0_cmd_bl),
    .c3_p0_cmd_byte_addr   (c3_p0_cmd_byte_addr),
    .c3_p0_cmd_empty       (c3_p0_cmd_empty),
    .c3_p0_cmd_full        (c3_p0_cmd_full),
    .c3_p0_wr_en           (c3_p0_wr_en),
    .c3_p0_wr_mask         (c3_p0_wr_mask),
    .c3_p0_wr_data         (c3_p0_wr_data),
    .c3_p0_wr_full         (c3_p0_wr_full),
    .c3_p0_wr_empty        (c3_p0_wr_empty),
    .c3_p0_wr_count        (c3_p0_wr_count),
    .c3_p0_wr_underrun     (c3_p0_wr_underrun),
    .c3_p0_wr_error        (c3_p0_wr_error),
    .c3_p0_rd_en           (c3_p0_rd_en),
    .c3_p0_rd_data         (c3_p0_rd_data),
    .c3_p0_rd_full         (c3_p0_rd_full),
    .c3_p0_rd_empty        (c3_p0_rd_empty),
    .c3_p0_rd_count        (c3_p0_rd_count),
    .c3_p0_rd_overflow     (c3_p0_rd_overflow),
    .c3_p0_rd_error        (c3_p0_rd_error)
  );

  // ========================================================================== //
  // Test-bench                                                                 //
  // ========================================================================== //

  localparam INSTR_WRITE           = 3'b000;
  localparam INSTR_READ            = 3'b001;
  localparam INSTR_WRITE_PRECHARGE = 3'b010;
  localparam INSTR_READ_PRECHARGE  = 3'b011;
  localparam INSTR_REFRESH         = 3'b100;

  reg [C3_P0_DATA_PORT_SIZE-1:0] input_data  [63:0];
  reg [C3_P0_DATA_PORT_SIZE-1:0] output_data [63:0];
  reg run_test_trigger = 0;

  initial begin
    c3_p0_cmd_en = 1'b0;
    c3_p0_cmd_instr = INSTR_WRITE;
    c3_p0_cmd_bl = 6'b000000;
    c3_p0_cmd_byte_addr = 30'b0;
    c3_p0_wr_en = 1'b0;
    c3_p0_wr_data = {C3_P0_DATA_PORT_SIZE{1'b0}};
    c3_p0_wr_mask = {C3_P0_MASK_SIZE{1'b0}};
    c3_p0_rd_en = 1'b0;
  end

  task clear_input_output_data;
    integer i;
    begin
      for (i = 0; i < 64; i = i + 1) input_data[i] = 0;
      for (i = 0; i < 64; i = i + 1) output_data[i] = 0;
    end
  endtask

  //
  // Compare input and output data for a given length
  //
  task compare_input_output_data;
    input nwords;
    integer i, num_mismatches;
    begin
      num_mismatches = 0;
      for (i = 0; i < nwords; i = i + 1) begin
        if (input_data[i] != output_data[i]) begin
          num_mismatches = num_mismatches + 1;
          $display("!!! Data mismatch at offset %2d: expected=%x, actual=%x", i, input_data[i], output_data[i]);
        end
      end
      if (num_mismatches > 0)
        $display("==> Found %d data mismatches.", num_mismatches);
      else
        $display("==> No data mismatches found.");
    end
  endtask

  //
  // Use a walking zero or one pattern for the input data
  //
  task use_walking_pattern;
    input  b;
    reg    [C3_P0_DATA_PORT_SIZE-1:0] x;
    integer i;
    begin
      x = 1'b1 ^~ {C3_P0_DATA_PORT_SIZE{b}};
      for (i = 0; i < 64; i = i + 1) begin
        input_data[i] = x;
        x = {x[C3_P0_DATA_PORT_SIZE-2:0], x[C3_P0_DATA_PORT_SIZE-1]};
      end
    end
  endtask

  //
  // Write data of a given burst size to specified starting memory address
  //
  task write_data;
    input  [C3_P0_BYTE_ADDR_WIDTH-1:0] start_addr;
    input  [6:0] burst_size;
    reg    [6:0] count;
    reg    [5:0] byte_length;
    begin
      count = 0;
      byte_length = burst_size - 1;
      while (count < burst_size) begin
        @(posedge c3_clk0);
        c3_p0_wr_en = 1'b1;
        if (!c3_p0_wr_full) begin
          c3_p0_wr_data = input_data[count];
          count = count + 1;
        end
      end

      @(posedge c3_clk0) c3_p0_wr_en = 1'b0;

      wait (!c3_p0_cmd_full);
      @(posedge c3_clk0) {c3_p0_cmd_en, c3_p0_cmd_instr, c3_p0_cmd_byte_addr, c3_p0_cmd_bl} = {1'b1, INSTR_WRITE, start_addr, byte_length};
      @(posedge c3_clk0) c3_p0_cmd_en = 1'b0;
    end
  endtask

  //
  // Read data of a given burst size from specified starting memory address
  //
  task read_data;
    input  [C3_P0_BYTE_ADDR_WIDTH-1:0] start_addr;
    input  [6:0] burst_size;
    reg    [6:0] count;
    reg    [5:0] byte_length;
    begin
      byte_length = burst_size - 1;
      wait (!c3_p0_cmd_full && (64 - c3_p0_rd_count >= burst_size));
      @(posedge c3_clk0) {c3_p0_cmd_en, c3_p0_cmd_instr, c3_p0_cmd_byte_addr, c3_p0_cmd_bl} = {1'b1, INSTR_READ, start_addr, byte_length};
      @(posedge c3_clk0) c3_p0_cmd_en = 1'b0;

      count = 0;
      while (count < burst_size) begin
        @(posedge c3_clk0);
        c3_p0_rd_en = 1'b1;
        if (!c3_p0_rd_empty) begin
          output_data[count] = c3_p0_rd_data;
          count = count + 1;
        end
      end

      @(posedge c3_clk0) c3_p0_rd_en = 1'b0;
    end
  endtask

  test_parameters params();

  task test_memory;
    reg    [C3_P0_BYTE_ADDR_WIDTH-1:0] addr;
    integer i;
    begin
      case (params.DataPattern)
        1: begin
          $display("### Using walking zero pattern");
          use_walking_pattern(0);
        end
        2: begin
          $display("### Using walking one pattern");
          use_walking_pattern(1);
        end
      endcase
      $display("### Read/write test started");
      for (addr = params.StartAddress; addr < params.EndAddress; addr = addr + 64 * 16) begin
        write_data(addr, 64);
        $display("... Writing %x", addr);
      end
      for (addr = params.StartAddress; addr < params.EndAddress; addr = addr + 64 * 16) begin
        read_data(addr, 64);
        $display("... Reading %x", addr);
        compare_input_output_data(64);
      end
      $display("### Read/write test done");
      $stop;
    end

  endtask

  always @(posedge run_test_trigger)
    test_memory;

  //
  // Tests
  //
  initial begin
    clear_input_output_data;
    wait (c3_rst_done);
    $display("### System reset done");
    wait (c3_calib_done);
    $display("### Memory calibration done");
  end

endmodule
