library ieee;
use ieee.std_logic_1164.all;

entity simple_arp is
  port (
    ll_clock        : in  std_logic;
    ll_reset        : in  std_logic;
    rx_ll_data      : in  std_logic_vector(7 downto 0);
    rx_ll_sof_n     : in  std_logic;
    rx_ll_eof_n     : in  std_logic;
    rx_ll_src_rdy_n : in  std_logic;
    rx_ll_dst_rdy_n : out std_logic;
    tx_ll_data      : out std_logic_vector(7 downto 0);
    tx_ll_sof_n     : out std_logic;
    tx_ll_eof_n     : out std_logic;
    tx_ll_src_rdy_n : out std_logic;
    tx_ll_dst_rdy_n : in  std_logic
  );
end simple_arp;

architecture rtl of simple_arp is

  type rx_state_t is (
    RX_DST_MAC_0,
    RX_DST_MAC_1,
    RX_DST_MAC_2,
    RX_DST_MAC_3,
    RX_DST_MAC_4,
    RX_DST_MAC_5,

    RX_SRC_MAC_0,
    RX_SRC_MAC_1,
    RX_SRC_MAC_2,
    RX_SRC_MAC_3,
    RX_SRC_MAC_4,
    RX_SRC_MAC_5,

    RX_ETH_TYPE_0,
    RX_ETH_TYPE_1,

    RX_ARP_HW_TYPE_0,
    RX_ARP_HW_TYPE_1,

    RX_ARP_PROTO_TYPE_0,
    RX_ARP_PROTO_TYPE_1,

    RX_ARP_HW_ADDR_LEN,
    RX_ARP_PROTO_ADDR_LEN,

    RX_ARP_OPCODE_0,
    RX_ARP_OPCODE_1,

    RX_SENDER_HW_ADDR_0,
    RX_SENDER_HW_ADDR_1,
    RX_SENDER_HW_ADDR_2,
    RX_SENDER_HW_ADDR_3,
    RX_SENDER_HW_ADDR_4,
    RX_SENDER_HW_ADDR_5,

    RX_SENDER_PROTO_ADDR_0,
    RX_SENDER_PROTO_ADDR_1,
    RX_SENDER_PROTO_ADDR_2,
    RX_SENDER_PROTO_ADDR_3,

    RX_TARGET_HW_ADDR_0,
    RX_TARGET_HW_ADDR_1,
    RX_TARGET_HW_ADDR_2,
    RX_TARGET_HW_ADDR_3,
    RX_TARGET_HW_ADDR_4,
    RX_TARGET_HW_ADDR_5,

    RX_TARGET_PROTO_ADDR_0,
    RX_TARGET_PROTO_ADDR_1,
    RX_TARGET_PROTO_ADDR_2,
    RX_TARGET_PROTO_ADDR_3,

    RX_SKIP_TO_EOF,
    RX_DONE,
    RX_WAIT_TX,
    RX_DISCARDED
  );

  type tx_state_t is (
    TX_DST_MAC_0,
    TX_DST_MAC_1,
    TX_DST_MAC_2,
    TX_DST_MAC_3,
    TX_DST_MAC_4,
    TX_DST_MAC_5,

    TX_SRC_MAC_0,
    TX_SRC_MAC_1,
    TX_SRC_MAC_2,
    TX_SRC_MAC_3,
    TX_SRC_MAC_4,
    TX_SRC_MAC_5,

    TX_ETH_TYPE_0,
    TX_ETH_TYPE_1,

    TX_ARP_HW_TYPE_0,
    TX_ARP_HW_TYPE_1,

    TX_ARP_PROTO_TYPE_0,
    TX_ARP_PROTO_TYPE_1,

    TX_ARP_HW_ADDR_LEN,
    TX_ARP_PROTO_ADDR_LEN,

    TX_ARP_OPCODE_0,
    TX_ARP_OPCODE_1,

    TX_SENDER_HW_ADDR_0,
    TX_SENDER_HW_ADDR_1,
    TX_SENDER_HW_ADDR_2,
    TX_SENDER_HW_ADDR_3,
    TX_SENDER_HW_ADDR_4,
    TX_SENDER_HW_ADDR_5,

    TX_SENDER_PROTO_ADDR_0,
    TX_SENDER_PROTO_ADDR_1,
    TX_SENDER_PROTO_ADDR_2,
    TX_SENDER_PROTO_ADDR_3,

    TX_TARGET_HW_ADDR_0,
    TX_TARGET_HW_ADDR_1,
    TX_TARGET_HW_ADDR_2,
    TX_TARGET_HW_ADDR_3,
    TX_TARGET_HW_ADDR_4,
    TX_TARGET_HW_ADDR_5,

    TX_TARGET_PROTO_ADDR_0,
    TX_TARGET_PROTO_ADDR_1,
    TX_TARGET_PROTO_ADDR_2,
    TX_TARGET_PROTO_ADDR_3,

    TX_DONE,
    TX_WAIT_RX
  );

  signal rx_state_cs : rx_state_t := RX_DST_MAC_0;
  signal rx_state_ns : rx_state_t;

  signal tx_state_cs : tx_state_t := TX_WAIT_RX;
  signal tx_state_ns : tx_state_t;

  type mac_addr_t is array (0 to 5) of std_logic_vector(7 downto 0);
  type ip_addr_t is array (0 to 3) of std_logic_vector(7 downto 0);

  signal src_mac_int : mac_addr_t;
  signal src_ip_int : ip_addr_t;

  signal dst_mac_cmb : mac_addr_t;
  signal dst_mac_int : mac_addr_t := (others => x"00");

  signal dst_ip_cmb : ip_addr_t;
  signal dst_ip_int : ip_addr_t := (others => x"00");

  signal rx_done_cmb : std_logic;
  signal rx_done_int : std_logic := '0';

  signal tx_done_cmb : std_logic;
  signal tx_done_int : std_logic := '0';

  signal rx_ll_dst_rdy_n_int : std_logic := '1';
  signal rx_ll_stable : boolean;
  signal tx_ll_src_rdy_n_int : std_logic := '1';
  signal tx_ll_stable : boolean;

begin
  src_mac_int(0) <= x"00";
  src_mac_int(1) <= x"0a";
  src_mac_int(2) <= x"35";
  src_mac_int(3) <= x"11";
  src_mac_int(4) <= x"22";
  src_mac_int(5) <= x"33";
  src_ip_int(0) <= x"c0";
  src_ip_int(1) <= x"a8";
  src_ip_int(2) <= x"08";
  src_ip_int(3) <= x"01";

  rx_ll_dst_rdy_n <= rx_ll_dst_rdy_n_int;
  rx_ll_stable <= rx_ll_src_rdy_n = '0' and rx_ll_dst_rdy_n_int = '0';

  process (rx_state_cs, rx_ll_stable, rx_ll_data, rx_ll_sof_n, rx_ll_eof_n, rx_done_int, tx_done_int, dst_mac_int, dst_ip_int)
  begin
    rx_state_ns <= rx_state_cs;
    rx_ll_dst_rdy_n_int <= '0';
    dst_mac_cmb <= dst_mac_int;
    dst_ip_cmb <= dst_ip_int;
    rx_done_cmb <= rx_done_int;

    case rx_state_cs is
      when RX_DST_MAC_0 =>
        if rx_ll_stable then
          if rx_ll_data = x"ff" and rx_ll_sof_n = '0' then
            rx_state_ns <= RX_DST_MAC_1;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_DST_MAC_1 =>
        if rx_ll_stable then
          if rx_ll_data = x"ff" then
            rx_state_ns <= RX_DST_MAC_2;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_DST_MAC_2 =>
        if rx_ll_stable then
          if rx_ll_data = x"ff" then
            rx_state_ns <= RX_DST_MAC_3;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_DST_MAC_3 =>
        if rx_ll_stable then
          if rx_ll_data = x"ff" then
            rx_state_ns <= RX_DST_MAC_4;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_DST_MAC_4 =>
        if rx_ll_stable then
          if rx_ll_data = x"ff" then
            rx_state_ns <= RX_DST_MAC_5;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_DST_MAC_5 =>
        if rx_ll_stable then
          if rx_ll_data = x"ff" then
            rx_state_ns <= RX_SRC_MAC_0;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;

      when RX_SRC_MAC_0 =>
        if rx_ll_stable then
          rx_state_ns <= RX_SRC_MAC_1;
          dst_mac_cmb(0) <= rx_ll_data;
        end if;
      when RX_SRC_MAC_1 =>
        if rx_ll_stable then
          rx_state_ns <= RX_SRC_MAC_2;
          dst_mac_cmb(1) <= rx_ll_data;
        end if;
      when RX_SRC_MAC_2 =>
        if rx_ll_stable then
          rx_state_ns <= RX_SRC_MAC_3;
          dst_mac_cmb(2) <= rx_ll_data;
        end if;
      when RX_SRC_MAC_3 =>
        if rx_ll_stable then
          rx_state_ns <= RX_SRC_MAC_4;
          dst_mac_cmb(3) <= rx_ll_data;
        end if;
      when RX_SRC_MAC_4 =>
        if rx_ll_stable then
          rx_state_ns <= RX_SRC_MAC_5;
          dst_mac_cmb(4) <= rx_ll_data;
        end if;
      when RX_SRC_MAC_5 =>
        if rx_ll_stable then
          rx_state_ns <= RX_ETH_TYPE_0;
          dst_mac_cmb(5) <= rx_ll_data;
        end if;

      when RX_ETH_TYPE_0 =>
        if rx_ll_stable then
          if rx_ll_data = x"08" then
            rx_state_ns <= RX_ETH_TYPE_1;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_ETH_TYPE_1 =>
        if rx_ll_stable then
          if rx_ll_data = x"06" then
            rx_state_ns <= RX_ARP_HW_TYPE_0;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;

      when RX_ARP_HW_TYPE_0 =>
        if rx_ll_stable then
          if rx_ll_data = x"00" then
            rx_state_ns <= RX_ARP_HW_TYPE_1;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_ARP_HW_TYPE_1 =>
        if rx_ll_stable then
          if rx_ll_data = x"01" then
            rx_state_ns <= RX_ARP_PROTO_TYPE_0;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;

      when RX_ARP_PROTO_TYPE_0 =>
        if rx_ll_stable then
          if rx_ll_data = x"08" then
            rx_state_ns <= RX_ARP_PROTO_TYPE_1;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_ARP_PROTO_TYPE_1 =>
        if rx_ll_stable then
          if rx_ll_data = x"00" then
            rx_state_ns <= RX_ARP_HW_ADDR_LEN;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;

      when RX_ARP_HW_ADDR_LEN =>
        if rx_ll_stable then
          if rx_ll_data = x"06" then
            rx_state_ns <= RX_ARP_PROTO_ADDR_LEN;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_ARP_PROTO_ADDR_LEN =>
        if rx_ll_stable then
          if rx_ll_data = x"04" then
            rx_state_ns <= RX_ARP_OPCODE_0;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;

      when RX_ARP_OPCODE_0 =>
        if rx_ll_stable then
          if rx_ll_data = x"00" then
            rx_state_ns <= RX_ARP_OPCODE_1;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_ARP_OPCODE_1 =>
        if rx_ll_stable then
          if rx_ll_data = x"01" then
            rx_state_ns <= RX_SENDER_HW_ADDR_0;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;

      when RX_SENDER_HW_ADDR_0 =>
        if rx_ll_stable then
          if rx_ll_data = dst_mac_int(0) then
            rx_state_ns <= RX_SENDER_HW_ADDR_1;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_SENDER_HW_ADDR_1 =>
        if rx_ll_stable then
          if rx_ll_data = dst_mac_int(1) then
            rx_state_ns <= RX_SENDER_HW_ADDR_2;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_SENDER_HW_ADDR_2 =>
        if rx_ll_stable then
          if rx_ll_data = dst_mac_int(2) then
            rx_state_ns <= RX_SENDER_HW_ADDR_3;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_SENDER_HW_ADDR_3 =>
        if rx_ll_stable then
          if rx_ll_data = dst_mac_int(3) then
            rx_state_ns <= RX_SENDER_HW_ADDR_4;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_SENDER_HW_ADDR_4 =>
        if rx_ll_stable then
          if rx_ll_data = dst_mac_int(4) then
            rx_state_ns <= RX_SENDER_HW_ADDR_5;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_SENDER_HW_ADDR_5 =>
        if rx_ll_stable then
          if rx_ll_data = dst_mac_int(5) then
            rx_state_ns <= RX_SENDER_PROTO_ADDR_0;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;

      when RX_SENDER_PROTO_ADDR_0 =>
        if rx_ll_stable then
          rx_state_ns <= RX_SENDER_PROTO_ADDR_1;
          dst_ip_cmb(0) <= rx_ll_data;
        end if;
      when RX_SENDER_PROTO_ADDR_1 =>
        if rx_ll_stable then
          rx_state_ns <= RX_SENDER_PROTO_ADDR_2;
          dst_ip_cmb(1) <= rx_ll_data;
        end if;
      when RX_SENDER_PROTO_ADDR_2 =>
        if rx_ll_stable then
          rx_state_ns <= RX_SENDER_PROTO_ADDR_3;
          dst_ip_cmb(2) <= rx_ll_data;
        end if;
      when RX_SENDER_PROTO_ADDR_3 =>
        if rx_ll_stable then
          rx_state_ns <= RX_TARGET_HW_ADDR_0;
          dst_ip_cmb(3) <= rx_ll_data;
        end if;

      when RX_TARGET_HW_ADDR_0 =>
        if rx_ll_stable then
          if rx_ll_data = x"00" then
            rx_state_ns <= RX_TARGET_HW_ADDR_1;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_TARGET_HW_ADDR_1 =>
        if rx_ll_stable then
          if rx_ll_data = x"00" then
            rx_state_ns <= RX_TARGET_HW_ADDR_2;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_TARGET_HW_ADDR_2 =>
        if rx_ll_stable then
          if rx_ll_data = x"00" then
            rx_state_ns <= RX_TARGET_HW_ADDR_3;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_TARGET_HW_ADDR_3 =>
        if rx_ll_stable then
          if rx_ll_data = x"00" then
            rx_state_ns <= RX_TARGET_HW_ADDR_4;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_TARGET_HW_ADDR_4 =>
        if rx_ll_stable then
          if rx_ll_data = x"00" then
            rx_state_ns <= RX_TARGET_HW_ADDR_5;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_TARGET_HW_ADDR_5 =>
        if rx_ll_stable then
          if rx_ll_data = x"00" then
            rx_state_ns <= RX_TARGET_PROTO_ADDR_0;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;

      when RX_TARGET_PROTO_ADDR_0 =>
        if rx_ll_stable then
          if rx_ll_data = src_ip_int(0) then
            rx_state_ns <= RX_TARGET_PROTO_ADDR_1;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_TARGET_PROTO_ADDR_1 =>
        if rx_ll_stable then
          if rx_ll_data = src_ip_int(1) then
            rx_state_ns <= RX_TARGET_PROTO_ADDR_2;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_TARGET_PROTO_ADDR_2 =>
        if rx_ll_stable then
          if rx_ll_data = src_ip_int(2) then
            rx_state_ns <= RX_TARGET_PROTO_ADDR_3;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;
      when RX_TARGET_PROTO_ADDR_3 =>
        if rx_ll_stable then
          if rx_ll_data = src_ip_int(3) then
            if rx_ll_eof_n = '0' then
              rx_done_cmb <= '1';
              rx_state_ns <= RX_DONE;
            else
              rx_state_ns <= RX_SKIP_TO_EOF;
            end if;
          else
            rx_state_ns <= RX_DISCARDED;
          end if;
        end if;

      when RX_SKIP_TO_EOF =>
        if rx_ll_stable then
          if rx_ll_eof_n = '0' then
            rx_done_cmb <= '1';
            rx_state_ns <= RX_DONE;
          end if;
        end if;

      when RX_DONE =>
        rx_ll_dst_rdy_n_int <= '1';
        rx_state_ns <= RX_WAIT_TX;

      when RX_WAIT_TX =>
        rx_ll_dst_rdy_n_int <= '1';
        if tx_done_int = '1' then
          rx_done_cmb <= '0';
          rx_state_ns <= RX_DST_MAC_0;
        end if;

      when RX_DISCARDED =>
        if rx_ll_stable then
          if rx_ll_eof_n = '0' then
            rx_state_ns <= RX_DST_MAC_0;
          end if;
        end if;

    end case;
  end process;

  process (ll_clock)
  begin
    if rising_edge(ll_clock) then
      if ll_reset = '1' then
        rx_state_cs <= RX_DST_MAC_0;
        rx_done_int <= '0';
        dst_mac_int <= (others => x"00");
        dst_ip_int <= (others => x"00");
      else
        rx_state_cs <= rx_state_ns;
        rx_done_int <= rx_done_cmb;
        dst_mac_int <= dst_mac_cmb;
        dst_ip_int <= dst_ip_cmb;
      end if;
    end if;
  end process;

  tx_ll_src_rdy_n <= tx_ll_src_rdy_n_int;
  tx_ll_stable <= tx_ll_src_rdy_n_int = '0' and tx_ll_dst_rdy_n = '0';

  process (tx_state_cs, tx_ll_stable, dst_mac_int, dst_ip_int, src_mac_int, src_ip_int, rx_done_int, tx_done_int)
  begin
    tx_state_ns <= tx_state_cs;
    tx_ll_src_rdy_n_int <= '0';
    tx_ll_sof_n <= '1';
    tx_ll_eof_n <= '1';
    tx_ll_data <= x"ff";
    tx_done_cmb <= tx_done_int;

    case tx_state_cs is

      when TX_DST_MAC_0 =>
        tx_ll_sof_n <= '0';
        tx_ll_data <= dst_mac_int(0);
        if tx_ll_stable then
          tx_state_ns <= TX_DST_MAC_1;
        end if;
      when TX_DST_MAC_1 =>
        tx_ll_data <= dst_mac_int(1);
        if tx_ll_stable then
          tx_state_ns <= TX_DST_MAC_2;
        end if;
      when TX_DST_MAC_2 =>
        tx_ll_data <= dst_mac_int(2);
        if tx_ll_stable then
          tx_state_ns <= TX_DST_MAC_3;
        end if;
      when TX_DST_MAC_3 =>
        tx_ll_data <= dst_mac_int(3);
        if tx_ll_stable then
          tx_state_ns <= TX_DST_MAC_4;
        end if;
      when TX_DST_MAC_4 =>
        tx_ll_data <= dst_mac_int(4);
        if tx_ll_stable then
          tx_state_ns <= TX_DST_MAC_5;
        end if;
      when TX_DST_MAC_5 =>
        tx_ll_data <= dst_mac_int(5);
        if tx_ll_stable then
          tx_state_ns <= TX_SRC_MAC_0;
        end if;

      when TX_SRC_MAC_0 =>
        tx_ll_data <= src_mac_int(0);
        if tx_ll_stable then
          tx_state_ns <= TX_SRC_MAC_1;
        end if;
      when TX_SRC_MAC_1 =>
        tx_ll_data <= src_mac_int(1);
        if tx_ll_stable then
          tx_state_ns <= TX_SRC_MAC_2;
        end if;
      when TX_SRC_MAC_2 =>
        tx_ll_data <= src_mac_int(2);
        if tx_ll_stable then
          tx_state_ns <= TX_SRC_MAC_3;
        end if;
      when TX_SRC_MAC_3 =>
        tx_ll_data <= src_mac_int(3);
        if tx_ll_stable then
          tx_state_ns <= TX_SRC_MAC_4;
        end if;
      when TX_SRC_MAC_4 =>
        tx_ll_data <= src_mac_int(4);
        if tx_ll_stable then
          tx_state_ns <= TX_SRC_MAC_5;
        end if;
      when TX_SRC_MAC_5 =>
        tx_ll_data <= src_mac_int(5);
        if tx_ll_stable then
          tx_state_ns <= TX_ETH_TYPE_0;
        end if;

      when TX_ETH_TYPE_0 =>
        tx_ll_data <= x"08";
        if tx_ll_stable then
          tx_state_ns <= TX_ETH_TYPE_1;
        end if;
      when TX_ETH_TYPE_1 =>
        tx_ll_data <= x"06";
        if tx_ll_stable then
          tx_state_ns <= TX_ARP_HW_TYPE_0;
        end if;

      when TX_ARP_HW_TYPE_0 =>
        tx_ll_data <= x"00";
        if tx_ll_stable then
          tx_state_ns <= TX_ARP_HW_TYPE_1;
        end if;
      when TX_ARP_HW_TYPE_1 =>
        tx_ll_data <= x"01";
        if tx_ll_stable then
          tx_state_ns <= TX_ARP_PROTO_TYPE_0;
        end if;

      when TX_ARP_PROTO_TYPE_0 =>
        tx_ll_data <= x"08";
        if tx_ll_stable then
          tx_state_ns <= TX_ARP_PROTO_TYPE_1;
        end if;
      when TX_ARP_PROTO_TYPE_1 =>
        tx_ll_data <= x"00";
        if tx_ll_stable then
          tx_state_ns <= TX_ARP_HW_ADDR_LEN;
        end if;

      when TX_ARP_HW_ADDR_LEN =>
        tx_ll_data <= x"06";
        if tx_ll_stable then
          tx_state_ns <= TX_ARP_PROTO_ADDR_LEN;
        end if;
      when TX_ARP_PROTO_ADDR_LEN =>
        tx_ll_data <= x"04";
        if tx_ll_stable then
          tx_state_ns <= TX_ARP_OPCODE_0;
        end if;

      when TX_ARP_OPCODE_0 =>
        tx_ll_data <= x"00";
        if tx_ll_stable then
          tx_state_ns <= TX_ARP_OPCODE_1;
        end if;
      when TX_ARP_OPCODE_1 =>
        tx_ll_data <= x"02";
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_HW_ADDR_0;
        end if;

      when TX_SENDER_HW_ADDR_0 =>
        tx_ll_data <= src_mac_int(0);
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_HW_ADDR_1;
        end if;
      when TX_SENDER_HW_ADDR_1 =>
        tx_ll_data <= src_mac_int(1);
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_HW_ADDR_2;
        end if;
      when TX_SENDER_HW_ADDR_2 =>
        tx_ll_data <= src_mac_int(2);
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_HW_ADDR_3;
        end if;
      when TX_SENDER_HW_ADDR_3 =>
        tx_ll_data <= src_mac_int(3);
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_HW_ADDR_4;
        end if;
      when TX_SENDER_HW_ADDR_4 =>
        tx_ll_data <= src_mac_int(4);
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_HW_ADDR_5;
        end if;
      when TX_SENDER_HW_ADDR_5 =>
        tx_ll_data <= src_mac_int(5);
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_PROTO_ADDR_0;
        end if;

      when TX_SENDER_PROTO_ADDR_0 =>
        tx_ll_data <= src_ip_int(0);
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_PROTO_ADDR_1;
        end if;
      when TX_SENDER_PROTO_ADDR_1 =>
        tx_ll_data <= src_ip_int(1);
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_PROTO_ADDR_2;
        end if;
      when TX_SENDER_PROTO_ADDR_2 =>
        tx_ll_data <= src_ip_int(2);
        if tx_ll_stable then
          tx_state_ns <= TX_SENDER_PROTO_ADDR_3;
        end if;
      when TX_SENDER_PROTO_ADDR_3 =>
        tx_ll_data <= src_ip_int(3);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_HW_ADDR_0;
        end if;

      when TX_TARGET_HW_ADDR_0 =>
        tx_ll_data <= dst_mac_int(0);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_HW_ADDR_1;
        end if;
      when TX_TARGET_HW_ADDR_1 =>
        tx_ll_data <= dst_mac_int(1);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_HW_ADDR_2;
        end if;
      when TX_TARGET_HW_ADDR_2 =>
        tx_ll_data <= dst_mac_int(2);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_HW_ADDR_3;
        end if;
      when TX_TARGET_HW_ADDR_3 =>
        tx_ll_data <= dst_mac_int(3);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_HW_ADDR_4;
        end if;
      when TX_TARGET_HW_ADDR_4 =>
        tx_ll_data <= dst_mac_int(4);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_HW_ADDR_5;
        end if;
      when TX_TARGET_HW_ADDR_5 =>
        tx_ll_data <= dst_mac_int(5);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_PROTO_ADDR_0;
        end if;

      when TX_TARGET_PROTO_ADDR_0 =>
        tx_ll_data <= dst_ip_int(0);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_PROTO_ADDR_1;
        end if;
      when TX_TARGET_PROTO_ADDR_1 =>
        tx_ll_data <= dst_ip_int(1);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_PROTO_ADDR_2;
        end if;
      when TX_TARGET_PROTO_ADDR_2 =>
        tx_ll_data <= dst_ip_int(2);
        if tx_ll_stable then
          tx_state_ns <= TX_TARGET_PROTO_ADDR_3;
        end if;
      when TX_TARGET_PROTO_ADDR_3 =>
        tx_ll_data <= dst_ip_int(3);
        tx_ll_eof_n <= '0';
        if tx_ll_stable then
          tx_state_ns <= TX_DONE;
          tx_done_cmb <= '1';
        end if;

      when TX_DONE =>
        tx_ll_src_rdy_n_int <= '1';
        tx_state_ns <= TX_WAIT_RX;

      when TX_WAIT_RX =>
        tx_ll_src_rdy_n_int <= '1';
        if rx_done_int = '1' then
          tx_done_cmb <= '0';
          tx_state_ns <= TX_DST_MAC_0;
        end if;

    end case;
  end process;

  process (ll_clock)
  begin
    if rising_edge(ll_clock) then
      if ll_reset = '1' then
        tx_state_cs <= TX_WAIT_RX;
        tx_done_int <= '0';
      else
        tx_state_cs <= tx_state_ns;
        tx_done_int <= tx_done_cmb;
      end if;
    end if;
  end process;

end rtl;
