library ieee;
use ieee.std_logic_1164.all;
use std.textio.all ;

entity v5emac_tb is
end v5emac_tb;

architecture testbench of v5emac_tb is

  component v5emac_top is
     port (
        clk              : in  std_logic;
        reset            : in  std_logic;
        resetdone        : out std_logic;

        ingress_sof_n    : out std_logic;
        ingress_eof_n    : out std_logic;
        ingress_data     : out std_logic_vector(7 downto 0);
        ingress_rd_count : out std_logic_vector(13 downto 0);
        ingress_re       : in  std_logic;
        ingress_empty    : out std_logic;

        egress_sof_n     : in  std_logic;
        egress_eof_n     : in  std_logic;
        egress_data      : in  std_logic_vector(7 downto 0);
        egress_wr_count  : out std_logic_vector(13 downto 0);
        egress_we        : in  std_logic;
        egress_full      : out std_logic;

        count_we_o       : out std_logic;
        count_wr_count   : out std_logic_vector(4 downto 0);

        TXP_0            : out std_logic;
        TXN_0            : out std_logic;
        RXP_0            : in  std_logic;
        RXN_0            : in  std_logic;
        MGTCLK_P         : in  std_logic;
        MGTCLK_N         : in  std_logic;
        PHY_RST_N        : out std_logic
     );
  end component;

  component address_swap_module_8 is
    port (
      rx_ll_clock         : in  std_logic;
      rx_ll_reset         : in  std_logic;
      rx_ll_data_in       : in  std_logic_vector(7 downto 0);
      rx_ll_sof_in_n      : in  std_logic;
      rx_ll_eof_in_n      : in  std_logic;
      rx_ll_src_rdy_in_n  : in  std_logic;
      rx_ll_data_out      : out std_logic_vector(7 downto 0);
      rx_ll_sof_out_n     : out std_logic;
      rx_ll_eof_out_n     : out std_logic;
      rx_ll_src_rdy_out_n : out std_logic;
      rx_ll_dst_rdy_in_n  : in  std_logic
    );
  end component;

  component simple_arp is
    port (
      ll_clock        : in  std_logic;
      ll_reset        : in  std_logic;
      rx_ll_data      : in  std_logic_vector(7 downto 0);
      rx_ll_sof_n     : in  std_logic;
      rx_ll_eof_n     : in  std_logic;
      rx_ll_src_rdy_n : in  std_logic;
      rx_ll_dst_rdy_n : out std_logic;
      tx_ll_data      : out std_logic_vector(7 downto 0);
      tx_ll_sof_n     : out std_logic;
      tx_ll_eof_n     : out std_logic;
      tx_ll_src_rdy_n : out std_logic;
      tx_ll_dst_rdy_n : in  std_logic
    );
  end component;

  signal clk               : std_logic := '0';
  signal reset             : std_logic := '1';
  signal resetdone         : std_logic;

  signal ingress_sof_n     : std_logic;
  signal ingress_sof_n_reg : std_logic;
  signal ingress_eof_n     : std_logic;
  signal ingress_data      : std_logic_vector(7 downto 0);
  signal ingress_rd_count  : std_logic_vector(13 downto 0);
  signal ingress_re        : std_logic;
  signal ingress_empty     : std_logic;

  signal egress_sof_n      : std_logic;
  signal egress_eof_n      : std_logic;
  signal egress_eof_n_reg  : std_logic;
  signal egress_data       : std_logic_vector(7 downto 0);
  signal egress_wr_count   : std_logic_vector(13 downto 0);
  signal egress_we         : std_logic;
  signal egress_full       : std_logic;

  signal count_wr_count    : std_logic_vector(4 downto 0);
  signal count_we_o        : std_logic;

  signal rx_ll_data        : std_logic_vector(7 downto 0);
  signal rx_ll_sof_n       : std_logic;
  signal rx_ll_eof_n       : std_logic;
  signal rx_ll_src_rdy_n   : std_logic;
  signal rx_ll_dst_rdy_n   : std_logic;
  signal tx_ll_data        : std_logic_vector(7 downto 0);
  signal tx_ll_sof_n       : std_logic;
  signal tx_ll_eof_n       : std_logic;
  signal tx_ll_src_rdy_n   : std_logic;
  signal tx_ll_dst_rdy_n   : std_logic;

  signal TXP_0             : std_logic;
  signal TXN_0             : std_logic;
  signal RXP_0             : std_logic;
  signal RXN_0             : std_logic;
  signal MGTCLK_P          : std_logic;
  signal MGTCLK_N          : std_logic;
  signal PHY_RST_N         : std_logic;

  constant DO_ADDRESS_SWAP : boolean := false;
  constant DO_INITIAL_RESET : boolean := true;

begin

  process
  begin
    clk <= not clk;
    wait for 5 ns;
  end process;

  g_init_reset: if DO_INITIAL_RESET = true generate
    process
    begin
      reset <= '1';
      wait for 20 ns;
      reset <= '0';
      wait;
    end process;
  end generate;

  g_no_init_reset : if DO_ADDRESS_SWAP = false generate
    reset <= '0';
  end generate;

  RXP_0 <= 'Z';
  RXN_0 <= 'Z';
  MGTCLK_P <= 'Z';
  MGTCLK_N <= 'Z';

  process (clk)
    variable log_line : line;
  begin
    if rising_edge(clk) then
      ingress_sof_n_reg <= ingress_sof_n;
      egress_eof_n_reg <= egress_eof_n;
      if ingress_sof_n = '0' and ingress_sof_n_reg = '1' then
        write(log_line, "receive a packet at ");
        write(log_line, now, justified => left, unit => ns);
        writeline(output, log_line);
      end if;
      if egress_eof_n = '0' and egress_eof_n_reg = '1' then
        write(log_line, "sent a packet at ");
        write(log_line, now, justified => left, unit => ns);
        writeline(output, log_line);
      end if;
    end if;
  end process;

  v5emac_dut : v5emac_top
    port map (
      clk              => clk,
      reset            => reset,
      resetdone        => resetdone,

      ingress_sof_n    => ingress_sof_n,
      ingress_eof_n    => ingress_eof_n,
      ingress_data     => ingress_data,
      ingress_rd_count => ingress_rd_count,
      ingress_re       => ingress_re,
      ingress_empty    => ingress_empty,

      egress_sof_n     => egress_sof_n,
      egress_eof_n     => egress_eof_n,
      egress_data      => egress_data,
      egress_wr_count  => egress_wr_count,
      egress_we        => egress_we,
      egress_full      => egress_full,

      count_we_o       => count_we_o,
      count_wr_count   => count_wr_count,

      TXP_0            => TXP_0,
      TXN_0            => TXN_0,
      RXP_0            => RXP_0,
      RXN_0            => RXN_0,
      MGTCLK_P         => MGTCLK_P,
      MGTCLK_N         => MGTCLK_N,
      PHY_RST_N        => PHY_RST_N
   );

  rx_ll_data <= ingress_data;
  rx_ll_sof_n <= ingress_sof_n;
  rx_ll_eof_n <= ingress_eof_n;
  rx_ll_src_rdy_n <= '0' when ingress_empty = '0' else '1';
  ingress_re <= '1' when rx_ll_src_rdy_n = '0' and rx_ll_dst_rdy_n = '0' else '0';

  egress_data <= tx_ll_data;
  egress_sof_n <= tx_ll_sof_n;
  egress_eof_n <= tx_ll_eof_n;
  tx_ll_dst_rdy_n <= '0' when egress_full = '0' else '1';
  egress_we <= '1' when tx_ll_src_rdy_n = '0' and tx_ll_dst_rdy_n = '0' else '0';

  g_swap_module : if DO_ADDRESS_SWAP = true generate

    packet_processor : address_swap_module_8
      port map (
        rx_ll_clock         => clk,
        rx_ll_reset         => reset,
        rx_ll_data_in       => rx_ll_data,
        rx_ll_sof_in_n      => rx_ll_sof_n,
        rx_ll_eof_in_n      => rx_ll_eof_n,
        rx_ll_src_rdy_in_n  => rx_ll_src_rdy_n,
        rx_ll_data_out      => tx_ll_data,
        rx_ll_sof_out_n     => tx_ll_sof_n,
        rx_ll_eof_out_n     => tx_ll_eof_n,
        rx_ll_src_rdy_out_n => tx_ll_src_rdy_n,
        rx_ll_dst_rdy_in_n  => tx_ll_dst_rdy_n
      );

    rx_ll_dst_rdy_n <= rx_ll_src_rdy_n;

  end generate;

  g_arp_reply : if DO_ADDRESS_SWAP = false generate

    packet_processor : simple_arp
      port map (
        ll_clock        => clk,
        ll_reset        => reset,
        rx_ll_data      => rx_ll_data,
        rx_ll_sof_n     => rx_ll_sof_n,
        rx_ll_eof_n     => rx_ll_eof_n,
        rx_ll_src_rdy_n => rx_ll_src_rdy_n,
        rx_ll_dst_rdy_n => rx_ll_dst_rdy_n,
        tx_ll_data      => tx_ll_data,
        tx_ll_sof_n     => tx_ll_sof_n,
        tx_ll_eof_n     => tx_ll_eof_n,
        tx_ll_src_rdy_n => tx_ll_src_rdy_n,
        tx_ll_dst_rdy_n => tx_ll_dst_rdy_n
      );

  end generate;

end testbench;
