`timescale 1ns / 1ps

module fp_fft_tb ();
  localparam FFT_XN_DATA_WIDTH          = 32;
  localparam FFT_XK_DATA_WIDTH          = 32;
  localparam FFT_TRANSFORM_LENGTH       = 16384;
  localparam FFT_TRANSFORM_LENGTH_LOG_2 = 14;
  localparam STATE_INIT_FFT             = 5'b00001;
  localparam STATE_START_FFT            = 5'b00010;
  localparam STATE_COMPUTE_FFT          = 5'b00100;
  localparam STATE_FFT_UNLOADING        = 5'b01000;
  localparam STATE_FFT_DONE             = 5'b10000;

  reg                                       clk = 1'b0;
  reg                                       fft_start;
  wire   [(FFT_XN_DATA_WIDTH-1):0]          fft_xn_re;
  wire   [(FFT_XN_DATA_WIDTH-1):0]          fft_xn_im;
  wire                                      fft_rfd;
  wire   [(FFT_TRANSFORM_LENGTH_LOG_2-1):0] fft_xn_index;
  wire                                      fft_fwd_inv;
  reg                                       fft_fwd_inv_we;
  wire                                      fft_busy;
  wire                                      fft_edone;
  wire                                      fft_done;
  wire                                      fft_dv;
  wire   [(FFT_TRANSFORM_LENGTH_LOG_2-1):0] fft_xk_index;
  wire   [(FFT_XK_DATA_WIDTH-1):0]          fft_xk_re;
  wire   [(FFT_XK_DATA_WIDTH-1):0]          fft_xk_im;

  reg    [(FFT_XN_DATA_WIDTH-1):0]          fft_xn_re_data [0:(FFT_TRANSFORM_LENGTH-1)];
  reg    [(FFT_XN_DATA_WIDTH-1):0]          fft_xn_im_data [0:(FFT_TRANSFORM_LENGTH-1)];
  reg    [(FFT_XK_DATA_WIDTH-1):0]          fft_xk_re_data [0:(FFT_TRANSFORM_LENGTH-1)];
  reg    [(FFT_XK_DATA_WIDTH-1):0]          fft_xk_im_data [0:(FFT_TRANSFORM_LENGTH-1)];
  reg    [(FFT_TRANSFORM_LENGTH_LOG_2-1):0] fft_xn_count     = 0;
  reg    [(FFT_TRANSFORM_LENGTH_LOG_2-1):0] fft_xn_count_reg = 0;
  reg    [(FFT_TRANSFORM_LENGTH_LOG_2-1):0] fft_xk_count     = 0;
  reg    [(FFT_TRANSFORM_LENGTH_LOG_2-1):0] fft_xk_count_reg = 0;
  reg    [4:0]                              current_state = STATE_INIT_FFT;
  reg    [4:0]                              next_state;


  function [31:0] reverse_vector;
    input [31:0] x;
    integer i;
    begin
      for (i = 0; i < 32; i = i + 1)
        reverse_vector[31-i] = x[i];
    end
  endfunction

  task clear_fft_xk_data;
    integer i;
    begin
      for (i = 0; i < FFT_TRANSFORM_LENGTH; i = i + 1) begin
        fft_xk_re_data[i] = 0;
        fft_xk_im_data[i] = 0;
      end
    end
  endtask

  task load_fft_xn_data;
    integer fid, i, status;
    begin
      fid = $fopen("fft_xn_data.txt", "r");
      for (i = 0; i < FFT_TRANSFORM_LENGTH; i = i + 1) begin
        status = $fscanf(fid, "%h %h\n", fft_xn_re_data[i], fft_xn_im_data[i]);
      end
      $fclose(fid);
    end
  endtask

  task save_fft_xk_data;
    integer fid, i;
    begin
      fid = $fopen("fft_xk_data.txt", "w");
      for (i = 0; i < FFT_TRANSFORM_LENGTH; i = i + 1) begin
        $fdisplay(fid, "%h %h", fft_xk_re_data[i], fft_xk_im_data[i]);
      end
      $fclose(fid);
    end
  endtask

  initial
  begin
    load_fft_xn_data;
    clear_fft_xk_data;
  end

  assign fft_fwd_inv = 1'b1;

  always
    #5 clk = ~clk;

  always @(posedge clk)
  begin
    current_state <= next_state;
    fft_xn_count_reg = fft_xn_count;
    fft_xk_count_reg = fft_xk_count;
  end

  always @(*)
  begin
    next_state = current_state;
    fft_fwd_inv_we = 1'b0;
    fft_start = 1'b0;
    fft_xn_count = fft_xn_count_reg;
    fft_xk_count = fft_xk_count_reg;

    case (current_state)
      STATE_INIT_FFT: begin
        fft_fwd_inv_we = 1'b1;
        next_state = STATE_START_FFT;
      end

      STATE_START_FFT: begin
        fft_start = 1'b1;
        fft_xn_count = 0;
        fft_xk_count = 0;
        next_state = STATE_COMPUTE_FFT;
      end

      STATE_COMPUTE_FFT: begin
        fft_start = 1'b1;
        if (fft_rfd) begin
          fft_xn_count = fft_xn_count_reg + 1;
          if (fft_xn_count_reg == 14'h3fff) begin
            fft_start = 1'b0;
            next_state = STATE_FFT_UNLOADING;
          end
        end
      end

      STATE_FFT_UNLOADING: begin
        if (fft_dv) begin
          fft_xk_count = fft_xk_count_reg + 1;
          if (fft_xk_count_reg == 14'h3fff)
            next_state = STATE_FFT_DONE;
        end
      end

      STATE_FFT_DONE: begin
        $display("### FFT Done");
        save_fft_xk_data;
        $finish;
      end
    endcase
  end

  assign fft_xn_re = fft_xn_re_data[fft_xn_index];
  assign fft_xn_im = fft_xn_im_data[fft_xn_index];

  always @(posedge clk)
  begin
    if (fft_dv) begin
      fft_xk_re_data[fft_xk_index] = reverse_vector(fft_xk_re);
      fft_xk_im_data[fft_xk_index] = reverse_vector(fft_xk_im);
    end
  end

  fp_fft_top dut (
    .clk            ( clk ),
    .fft_start      ( fft_start ),
    .fft_xn_re      ( fft_xn_re ),
    .fft_xn_im      ( fft_xn_im ),
    .fft_fwd_inv    ( fft_fwd_inv ),
    .fft_fwd_inv_we ( fft_fwd_inv_we ),
    .fft_rfd        ( fft_rfd ),
    .fft_xn_index   ( fft_xn_index ),
    .fft_busy       ( fft_busy ),
    .fft_edone      ( fft_edone ),
    .fft_done       ( fft_done ),
    .fft_dv         ( fft_dv ),
    .fft_xk_index   ( fft_xk_index ),
    .fft_xk_re      ( fft_xk_re ),
    .fft_xk_im      ( fft_xk_im )
  );

endmodule
