library unisim;
use unisim.vcomponents.all;

library ieee;
use ieee.std_logic_1164.all;

entity v5emac_top is
  port(
    clk              : in  std_logic;
    reset            : in  std_logic;
    resetdone        : out std_logic;

    ingress_sof_n    : out std_logic;
    ingress_eof_n    : out std_logic;
    ingress_data     : out std_logic_vector(7 downto 0);
    ingress_rd_count : out std_logic_vector(13 downto 0);
    ingress_re       : in  std_logic;
    ingress_empty    : out std_logic;

    egress_sof_n     : in  std_logic;
    egress_eof_n     : in  std_logic;
    egress_data      : in  std_logic_vector(7 downto 0);
    egress_wr_count  : out std_logic_vector(13 downto 0);
    egress_we        : in  std_logic;
    egress_full      : out std_logic;

    count_we_o       : out std_logic;
    count_wr_count   : out std_logic_vector(4 downto 0);

    TXP_0            : out std_logic;
    TXN_0            : out std_logic;
    RXP_0            : in  std_logic;
    RXN_0            : in  std_logic;
    MGTCLK_P         : in  std_logic;
    MGTCLK_N         : in  std_logic;
    PHY_RST_N        : out std_logic
  );
end v5emac_top;

architecture rtl of v5emac_top is

  -- Component Declaration for the TEMAC wrapper with
  -- Local Link FIFO.
  component v5emac_locallink is
    port(
      -- EMAC0 Clocking
      -- 125MHz clock output from transceiver
      CLK125_OUT                       : out std_logic;
      -- 125MHz clock input from BUFG
      CLK125                           : in  std_logic;

      -- Local link Receiver Interface - EMAC0
      RX_LL_CLOCK_0                   : in  std_logic;
      RX_LL_RESET_0                   : in  std_logic;
      RX_LL_DATA_0                    : out std_logic_vector(7 downto 0);
      RX_LL_SOF_N_0                   : out std_logic;
      RX_LL_EOF_N_0                   : out std_logic;
      RX_LL_SRC_RDY_N_0               : out std_logic;
      RX_LL_DST_RDY_N_0               : in  std_logic;
      RX_LL_FIFO_STATUS_0             : out std_logic_vector(3 downto 0);

      -- Local link Transmitter Interface - EMAC0
      TX_LL_CLOCK_0                   : in  std_logic;
      TX_LL_RESET_0                   : in  std_logic;
      TX_LL_DATA_0                    : in  std_logic_vector(7 downto 0);
      TX_LL_SOF_N_0                   : in  std_logic;
      TX_LL_EOF_N_0                   : in  std_logic;
      TX_LL_SRC_RDY_N_0               : in  std_logic;
      TX_LL_DST_RDY_N_0               : out std_logic;

      -- Client Receiver Interface - EMAC0
      EMAC0CLIENTRXDVLD               : out std_logic;
      EMAC0CLIENTRXFRAMEDROP          : out std_logic;
      EMAC0CLIENTRXSTATS              : out std_logic_vector(6 downto 0);
      EMAC0CLIENTRXSTATSVLD           : out std_logic;
      EMAC0CLIENTRXSTATSBYTEVLD       : out std_logic;

      -- Client Transmitter Interface - EMAC0
      CLIENTEMAC0TXIFGDELAY           : in  std_logic_vector(7 downto 0);
      EMAC0CLIENTTXSTATS              : out std_logic;
      EMAC0CLIENTTXSTATSVLD           : out std_logic;
      EMAC0CLIENTTXSTATSBYTEVLD       : out std_logic;

      -- MAC Control Interface - EMAC0
      CLIENTEMAC0PAUSEREQ             : in  std_logic;
      CLIENTEMAC0PAUSEVAL             : in  std_logic_vector(15 downto 0);

      --EMAC-MGT link status
      EMAC0CLIENTSYNCACQSTATUS        : out std_logic;
      -- EMAC0 Interrupt
      EMAC0ANINTERRUPT                : out std_logic;

      -- Clock Signals - EMAC0

      -- SGMII Interface - EMAC0
      TXP_0                           : out std_logic;
      TXN_0                           : out std_logic;
      RXP_0                           : in  std_logic;
      RXN_0                           : in  std_logic;
      PHYAD_0                         : in  std_logic_vector(4 downto 0);
      RESETDONE_0                     : out std_logic;

      -- unused transceiver
      TXN_1_UNUSED                    : out std_logic;
      TXP_1_UNUSED                    : out std_logic;
      RXN_1_UNUSED                    : in  std_logic;
      RXP_1_UNUSED                    : in  std_logic;

      -- SGMII RocketIO Reference Clock buffer inputs
      CLK_DS                          : in  std_logic;

      -- RocketIO Reset input
      GTRESET                         : in  std_logic;

      -- Asynchronous Reset
      RESET                           : in  std_logic
    );
  end component;

  component packet_fifo
    port (
      din           : in std_logic_vector(9 downto 0);
      rd_clk        : in std_logic;
      rd_en         : in std_logic;
      rst           : in std_logic;
      wr_clk        : in std_logic;
      wr_en         : in std_logic;
      dout          : out std_logic_vector(9 downto 0);
      empty         : out std_logic;
      full          : out std_logic;
      rd_data_count : out std_logic_vector(13 downto 0);
      wr_data_count : out std_logic_vector(13 downto 0)
    );
  end component;

  component count_fifo
    port (
      din           : in  std_logic_vector(0 downto 0);
      rd_clk        : in  std_logic;
      rd_en         : in  std_logic;
      rst           : in  std_logic;
      wr_clk        : in  std_logic;
      wr_en         : in  std_logic;
      dout          : out std_logic_vector(0 downto 0);
      empty         : out std_logic;
      full          : out std_logic;
      rd_data_count : out std_logic_vector(4 downto 0);
      wr_data_count : out std_logic_vector(4 downto 0)
    );
  end component;

  -- Global asynchronous reset
  signal reset_i               : std_logic;
  signal resetdone_i           : std_logic;

  -- Transceiver output clock (REFCLKOUT at 125MHz)
  signal clk125_o              : std_logic;
  -- 125MHz clock input to wrappers
  signal clk125                : std_logic;
  -- Input 125MHz differential clock for transceiver
  signal clk_ds                : std_logic;

  -- GT reset signal
  signal gtreset               : std_logic;
  signal reset_r               : std_logic_vector(3 downto 0);

  signal ll_clock              : std_logic := '0';
  signal ll_reset              : std_logic := '1';
  signal ll_pre_reset          : std_logic_vector(5 downto 0) := (others => '1');

  signal rx_ll_data            : std_logic_vector(7 downto 0);
  signal rx_ll_sof_n           : std_logic;
  signal rx_ll_eof_n           : std_logic;
  signal rx_ll_src_rdy_n       : std_logic;
  signal rx_ll_dst_rdy_n       : std_logic;

  signal tx_ll_data            : std_logic_vector(7 downto 0);
  signal tx_ll_sof_n           : std_logic;
  signal tx_ll_eof_n           : std_logic;
  signal tx_ll_src_rdy_n       : std_logic;
  signal tx_ll_dst_rdy_n       : std_logic;
  type tx_ll_state_t is ( S_TX_LL_IDLE, S_TX_LL_SOF, S_TX_LL_IN_PROGRESS );
  signal tx_ll_state_cs        : tx_ll_state_t := S_TX_LL_IDLE;
  signal tx_ll_state_ns        : tx_ll_state_t;

  signal ingress_we            : std_logic;
  signal ingress_empty_i       : std_logic;
  signal ingress_full          : std_logic;
  signal ingress_wr_count      : std_logic_vector(13 downto 0);
  signal ingress_din           : std_logic_vector(9 downto 0);
  signal ingress_dout          : std_logic_vector(9 downto 0);
  type ingress_state_t is ( S_INGRESS_IDLE, S_INGRESS_IN_PROGRESS );
  signal ingress_state_cs      : ingress_state_t := S_INGRESS_IDLE;
  signal ingress_state_ns      : ingress_state_t;

  signal egress_re             : std_logic;
  signal egress_empty          : std_logic;
  signal egress_rd_count       : std_logic_vector(13 downto 0);
  signal egress_din            : std_logic_vector(9 downto 0);
  signal egress_dout           : std_logic_vector(9 downto 0);
  type egress_state_t is ( S_EGRESS_IDLE, S_EGRESS_IN_PROGRESS );
  signal egress_state_cs       : egress_state_t := S_EGRESS_IDLE;
  signal egress_state_ns       : egress_state_t;

  signal count_re              : std_logic;
  signal count_empty           : std_logic;
  signal count_we              : std_logic;

  attribute async_reg : string;
  attribute async_reg of ll_pre_reset : signal is "true";
  attribute async_reg of reset_r      : signal is "TRUE";

begin

  reset_i <= reset;
  resetdone <= resetdone_i;

  -- Generate the clock input to the GTP
  -- clk_ds can be shared between multiple MAC instances.
  clkingen : IBUFDS
    port map (
      I  => MGTCLK_P,
      IB => MGTCLK_N,
      O  => clk_ds
    );

  -- 125MHz from transceiver is routed through a BUFG and
  -- input to the MAC wrappers.
  -- This clock can be shared between multiple MAC instances.
  bufg_clk125 : BUFG
    port map (
      I => clk125_o,
      O => clk125
    );

  --------------------------------------------------------------------
  -- RocketIO PMA reset circuitry
  --------------------------------------------------------------------
  process(reset_i, clk125)
  begin
    if (reset_i = '1') then
      reset_r <= "1111";
    elsif clk125'event and clk125 = '1' then
      reset_r <= reset_r(2 downto 0) & reset_i;
    end if;
  end process;

  gtreset <= reset_r(3);

  obuf_ethernet_phy_rst_n : OBUF
    port map (
      I => '1',
      O => PHY_RST_N
    );

  ------------------------------------------------------------------------
  -- Instantiate the EMAC Wrapper with LL FIFO
  -- (v5emac_locallink.v)
  ------------------------------------------------------------------------
  v5_emac_ll : v5emac_locallink
    port map (
      -- EMAC0 Clocking
      -- 125MHz clock output from transceiver
      CLK125_OUT                      => clk125_o,
      -- 125MHz clock input from BUFG
      CLK125                          => clk125,
      -- Local link Receiver Interface - EMAC0
      RX_LL_CLOCK_0                   => ll_clock,
      RX_LL_RESET_0                   => ll_reset,
      RX_LL_DATA_0                    => rx_ll_data,
      RX_LL_SOF_N_0                   => rx_ll_sof_n,
      RX_LL_EOF_N_0                   => rx_ll_eof_n,
      RX_LL_SRC_RDY_N_0               => rx_ll_src_rdy_n,
      RX_LL_DST_RDY_N_0               => rx_ll_dst_rdy_n,
      RX_LL_FIFO_STATUS_0             => open,

      -- Unused Receiver signals - EMAC0
      EMAC0CLIENTRXDVLD               => open,
      EMAC0CLIENTRXFRAMEDROP          => open,
      EMAC0CLIENTRXSTATS              => open,
      EMAC0CLIENTRXSTATSVLD           => open,
      EMAC0CLIENTRXSTATSBYTEVLD       => open,

      -- Local link Transmitter Interface - EMAC0
      TX_LL_CLOCK_0                   => ll_clock,
      TX_LL_RESET_0                   => ll_reset,
      TX_LL_DATA_0                    => tx_ll_data,
      TX_LL_SOF_N_0                   => tx_ll_sof_n,
      TX_LL_EOF_N_0                   => tx_ll_eof_n,
      TX_LL_SRC_RDY_N_0               => tx_ll_src_rdy_n,
      TX_LL_DST_RDY_N_0               => tx_ll_dst_rdy_n,

      -- Unused Transmitter signals - EMAC0
      CLIENTEMAC0TXIFGDELAY           => (others => '0'),
      EMAC0CLIENTTXSTATS              => open,
      EMAC0CLIENTTXSTATSVLD           => open,
      EMAC0CLIENTTXSTATSBYTEVLD       => open,

      -- MAC Control Interface - EMAC0
      CLIENTEMAC0PAUSEREQ             => '0',
      CLIENTEMAC0PAUSEVAL             => (others => '0'),

      --EMAC-MGT link status
      EMAC0CLIENTSYNCACQSTATUS        => open,
      -- EMAC0 Interrupt
      EMAC0ANINTERRUPT                => open,

      -- Clock Signals - EMAC0
      -- SGMII Interface - EMAC0
      TXP_0                           => TXP_0,
      TXN_0                           => TXN_0,
      RXP_0                           => RXP_0,
      RXN_0                           => RXN_0,
      PHYAD_0                         => "00111",
      RESETDONE_0                     => resetdone_i,

      -- unused transceiver
      TXN_1_UNUSED                    => open,
      TXP_1_UNUSED                    => open,
      RXN_1_UNUSED                    => '0',
      RXP_1_UNUSED                    => '0',

      -- SGMII RocketIO Reference Clock buffer inputs
      CLK_DS                          => clk_ds,

      -- RocketIO Reset input
      GTRESET                         => gtreset,

      -- Asynchronous Reset
      RESET                           => reset_i
    );

  ------------------------------------------------------------------------

  -- Create synchronous reset in the transmitter clock domain.
  gen_ll_reset : process (ll_clock, reset)
  begin
    if reset = '1' then
      ll_pre_reset <= (others => '1');
      ll_reset     <= '1';
    elsif ll_clock'event and ll_clock = '1' then
      if resetdone_i = '1' then
        ll_pre_reset(0)          <= '0';
        ll_pre_reset(5 downto 1) <= ll_pre_reset(4 downto 0);
        ll_reset                 <= ll_pre_reset(5);
      end if;
    end if;
  end process;

  ll_clock <= clk125;

  ------------------------------------------------------------------------

  process (ingress_state_cs, ingress_empty_i, ingress_dout)
  begin
    ingress_state_ns <= ingress_state_cs;
    ingress_data <= (others => '1');
    ingress_eof_n <= '1';
    ingress_sof_n <= '1';
    ingress_empty <= '1';
    case ingress_state_cs is
      when S_INGRESS_IDLE =>
        if ingress_empty_i = '0' then
          ingress_state_ns <= S_INGRESS_IN_PROGRESS;
        end if;
      when S_INGRESS_IN_PROGRESS =>
        ingress_data <= ingress_dout(7 downto 0);
        ingress_eof_n <= ingress_dout(8);
        ingress_sof_n <= ingress_dout(9);
        ingress_empty <= ingress_empty_i;
        if ingress_dout(8) = '0' then
          ingress_state_ns <= S_INGRESS_IDLE;
        end if;
    end case;
  end process;

  process (clk)
  begin
    if reset = '1' then
      ingress_state_cs <= S_INGRESS_IDLE;
    elsif rising_edge(clk) then
      ingress_state_cs <= ingress_state_ns;
    end if;
  end process;

  ingress_din <= rx_ll_sof_n & rx_ll_eof_n & rx_ll_data;

  ingress_fifo : packet_fifo
    port map (
      rst           => reset,

      rd_clk        => clk,
      rd_en         => ingress_re,
      dout          => ingress_dout,
      empty         => ingress_empty_i,
      rd_data_count => ingress_rd_count,

      wr_clk        => ll_clock,
      wr_en         => ingress_we,
      din           => ingress_din,
      full          => ingress_full,
      wr_data_count => ingress_wr_count
    );

  process (ingress_full)
  begin
    rx_ll_dst_rdy_n <= '1';
    if ingress_full = '0' then
      rx_ll_dst_rdy_n <= '0';
    end if;
  end process;

  process (rx_ll_src_rdy_n, rx_ll_dst_rdy_n)
  begin
    ingress_we <= '0';
    if rx_ll_src_rdy_n = '0' and rx_ll_dst_rdy_n = '0' then
      ingress_we <= '1';
    end if;
  end process;

  ------------------------------------------------------------------------

  egress_din <= egress_sof_n & egress_eof_n & egress_data;

  egress_fifo : packet_fifo
    port map (
      rst           => reset,

      rd_clk        => ll_clock,
      rd_en         => egress_re,
      dout          => egress_dout,
      empty         => egress_empty,
      rd_data_count => egress_rd_count,

      wr_clk        => clk,
      wr_en         => egress_we,
      din           => egress_din,
      full          => egress_full,
      wr_data_count => egress_wr_count
    );

  count_fifo_inst : count_fifo
    port map (
      rst           => reset,

      rd_clk        => ll_clock,
      rd_en         => count_re,
      dout          => open,
      empty         => count_empty,
      rd_data_count => open,

      wr_clk        => clk,
      wr_en         => count_we,
      din           => "1",
      full          => open,
      wr_data_count => count_wr_count
    );

  count_we_o <= count_we;

  process (
    egress_state_cs, egress_we, egress_sof_n, egress_eof_n
  )
  begin
    egress_state_ns <= egress_state_cs;
    count_we <= '0';
    case egress_state_cs is
      when S_EGRESS_IDLE =>
        if egress_we = '1' and egress_sof_n = '0' then
          egress_state_ns <= S_EGRESS_IN_PROGRESS;
        end if;
      when S_EGRESS_IN_PROGRESS =>
        if egress_we = '1' and egress_eof_n = '0' then
          count_we <= '1';
          egress_state_ns <= S_EGRESS_IDLE;
        end if;
    end case;
  end process;

  process (clk)
  begin
    if reset = '1' then
      egress_state_cs <= S_EGRESS_IDLE;
    elsif rising_edge(clk) then
      egress_state_cs <= egress_state_ns;
    end if;
  end process;

  process (
    tx_ll_state_cs, egress_empty, egress_dout, count_empty,
    tx_ll_dst_rdy_n, tx_ll_sof_n, tx_ll_eof_n
  )
  begin
    tx_ll_state_ns <= tx_ll_state_cs;
    egress_re <= '0';
    count_re <= '0';
    tx_ll_data <= egress_dout(7 downto 0);
    tx_ll_eof_n <= egress_dout(8);
    tx_ll_sof_n <= egress_dout(9);
    tx_ll_src_rdy_n <= '1';

    case tx_ll_state_cs is
      when S_TX_LL_IDLE =>
        tx_ll_data <= (others => '1');
        tx_ll_eof_n <= '1';
        tx_ll_sof_n <= '1';
        if count_empty = '0' then
          count_re <= '1';
          tx_ll_state_ns <= S_TX_LL_SOF;
        end if;

      when S_TX_LL_SOF =>
        tx_ll_eof_n <= '1';
        if egress_empty = '0' then
          tx_ll_src_rdy_n <= '0';
          if tx_ll_dst_rdy_n = '0' then
            egress_re <= '1';
            if tx_ll_sof_n = '0' then
              tx_ll_state_ns <= S_TX_LL_IN_PROGRESS;
            end if;
          end if;
        end if;

      when S_TX_LL_IN_PROGRESS =>
        if egress_empty = '0' then
          tx_ll_src_rdy_n <= '0';
          if tx_ll_dst_rdy_n = '0' then
            egress_re <= '1';
            if tx_ll_eof_n = '0' then
              tx_ll_state_ns <= S_TX_LL_IDLE;
            end if;
          end if;
        end if;
    end case;
  end process;

  process (ll_clock)
  begin
    if rising_edge(ll_clock) then
      if ll_reset = '1' then
        tx_ll_state_cs <= S_TX_LL_IDLE;
      else
        tx_ll_state_cs <= tx_ll_state_ns;
      end if;
    end if;
  end process;

end rtl;
