----------------------------------------------------------------------------------
-- Company: Xilinx, Inc.
-- Engineer: Eddie Vergara
-- 
-- Create Date:    08:38:05 03/16/2009 
-- Design Name:    DRP Demo - Test Bench
-- Module Name:    drp_demo_tb - Behavioral 
-- Project Name:   DRP Demo
-- Target Devices: xc5vlx30
-- Tool versions:  11.1
-- Description:    The DRP Demo Test Bench provides the system clock (CLK_IN) and asserts the 
--                 DRP state machine start signals to change the output clock (DCM_CLKFX_OUT)
--                 frequency based on provided Multiplier/Divider values 
--
----------------------------------------------------------------------------------


library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

library STD;
use STD.TEXTIO.all;

library drp_tb_lib;
use drp_tb_lib.drp_tb_pkg.all;

ENTITY drp_demo_tb IS
END drp_demo_tb;

ARCHITECTURE behavioral OF drp_demo_tb IS

   component drp_demo 
    Port ( 
           --Clock and Reset--
    	   clk_in : in  STD_LOGIC;
           drp_stmach_reset : in STD_LOGIC;
	   user_dcm_reset : in STD_LOGIC;
	   
	   --DRP User Interface--
           drp_start : in  STD_LOGIC;
	   drp_multiply, drp_divide : in STD_LOGIC_VECTOR (0 to 7);
	   drp_change_mode : in STD_LOGIC;
	   drp_current_mode : in STD_LOGIC;

	   --DCM Outputs--
	   dcm_clk0_out : out STD_LOGIC;
           dcm_clkfx_out : out  STD_LOGIC;
	   
	   --Monitor Signals--
	   dcm_locked : out STD_LOGIC;
           drp_done : out STD_LOGIC);
	   
    end component;

    SIGNAL clk_in : std_logic := '0';
    
    SIGNAL drp_start, drp_change_mode, drp_current_mode, 
           drp_done, drp_stmach_reset, user_dcm_reset : std_logic := '0';

    SIGNAL drp_multiply, drp_divide : std_logic_vector (7 DownTo 0) := x"00";
    
    SIGNAL dcm_clk0_out, dcm_clkfx_out, dcm_locked : std_logic := '0';
    
    constant PERIOD : time := 10000 ps;


   
    BEGIN
        UUT : drp_demo
        PORT MAP (
            	clk_in => clk_in,
		drp_stmach_reset => drp_stmach_reset,
		user_dcm_reset => user_dcm_reset,
		drp_start => drp_start,
		drp_change_mode => drp_change_mode,
		drp_current_mode => drp_current_mode,
		drp_multiply => drp_multiply,
		drp_divide => drp_divide,
		dcm_clk0_out => dcm_clk0_out,
		dcm_clkfx_out => dcm_clkfx_out,
		dcm_locked => dcm_locked,
		drp_done => drp_done
        );

        PROCESS    -- clock process for clk_in
        BEGIN
            CLOCK_LOOP : LOOP
                clk_in <= '0';
                WAIT FOR PERIOD * 0.5;
                clk_in <= '1';
                WAIT FOR PERIOD * 0.5;
            END LOOP CLOCK_LOOP;
        END PROCESS;

        PROCESS
	
	variable Message : line;
	variable clkfx_edge_current, clkfx_edge_previous : time;
	
	type bool_array is array (integer range <>) of boolean;
	
	type vector_record is record 
		test_num: integer; -- Test Number
		freq: integer; -- Test Frequency
		period: time; -- Test Period
		multiplier: integer; -- Test Multiplier
		divider: integer;  -- Test Divider

	end record;

	type vector_array is array (integer range 1 to 4) of vector_record; 

	variable test_result : bool_array (1 to 4);

	    ------------------------------------------------
	    -- ** TEST VECTORS **
	    --  Test, Frequency, Period, Multiplier, Divider
	    ------------------------------------------------
	
	constant test_vectors : vector_array := (
	  
	  ( 1, 75, 13332 ps, 3, 4),
	  ( 2, 120, 8332 ps, 7, 5),
	  ( 3, 250, 4000 ps, 5, 2),
	  ( 4, 400, 2500 ps, 3, 1));
	
            BEGIN

	    -----------------------------
	    -- ** START OF SIMULATION **  
	    -----------------------------

	    ----------------------------------------------
	    -- System Reset
	    ----------------------------------------------

		Write ( Message, LF);
	    	Write ( Message, string'("-- Resets asserted ...")&LF);
		assert false report "" severity note;
		writeline(output, Message);

		drp_stmach_reset <= '1';
		user_dcm_reset <= '1';

		WAIT for 150 ns;

		drp_stmach_reset <= '0';
		user_dcm_reset <= '0';
		
	    	Write ( Message, LF);
		Write ( Message, string'("-- Resets deasserted ...")&LF);
		assert false report "" severity note;
		writeline(output, Message);

		WAIT for 1 us;

	    -----------------------------------------------------------------------
	    -- Test 1: Set CLKFX = 75 MHz (13332 ps) via DRP Cycle
	    -- Test 2: Set CLKFX = 120 MHz (8333 ps) via DRP Cycle
	    -----------------------------------------------------------------------
    
	    for i in 1 to 2
		loop 

		Write ( Message, LF);
		Write ( Message, string'("-- Test "));
		Write ( Message, test_vectors(i).test_num);
		Write ( Message, string'(": *START* - Set DCM_CLKFX_OUT = "));
		Write ( Message, test_vectors(i).freq);
		Write ( Message, string'(" MHz ("));
		Write ( Message, test_vectors(i).period);
		Write ( Message, string'(") via DRP Cycle (M="));
		Write ( Message, test_vectors(i).multiplier);
		Write ( Message, string'(", D="));
		Write ( Message, test_vectors(i).divider);
		Write ( Message, string'(")")&LF);

		assert false report "" severity note;
		writeline(output, Message);

	    -- 1. Set Multiplier and Divider values from test vector
				
		drp_multiply <= conv_std_logic_vector((test_vectors(i).multiplier - 1), 8);
		drp_divide <= conv_std_logic_vector((test_vectors(i).divider - 1), 8);
		
	    -- 2. Signal DRP Controller to start a DRP cycle

		drp_start <= '1';	     
		
   	    -- 3. Wait for DCM Lock, indicating end of DRP Cycle

		WAIT UNTIL drp_done = '1';   
		drp_start <= '0';	     

		Write ( Message, LF);
		Write ( Message, string'("-- Test "));
		Write ( Message, i);
		Write ( Message, string'(": Achieved DCM LOCK ...")&LF);
		assert false report "" severity note;
		writeline(output, Message);

	    -- 4. Read simulation time for two consequent DCM_CLKFX_OUT rising edges

		WAIT until (rising_edge(dcm_clkfx_out));
		clkfx_edge_previous := NOW;
				
		WAIT until (rising_edge(dcm_clkfx_out));
		clkfx_edge_current := NOW;

	    -- 5. Compare DCM_CLKFX_OUT actual period with expected period.

		test_result(i) := period_check (i, clkfx_edge_current, clkfx_edge_previous, test_vectors(i).period);

	    -- 6. End of Test

		WAIT for 2 us;

	end loop;
	    
	    -----------------------------------------------------------------------
	    -- Change DFS_FREQUENCY MODE from "LOW" to "HIGH" for High Speed test
	    -----------------------------------------------------------------------

		Write ( Message, LF);
		Write ( Message, string'("-- Changing DFS_FREQUENCY_MODE LOW --> HIGH ...")&LF);
		assert false report "" severity note;
		writeline(output, Message);

	     -- 1. Set DRP Controller signals for DFS_FREQUENCY_MODE = HIGH.

		drp_current_mode <= '1';
		drp_change_mode <= '1';

	     -- 2. Signal DRP Controller to start a DRP cycle
		drp_start <= '1';

	     -- 3. Wait for DCM Lock, indicating end of DRP Cycle

		WAIT UNTIL drp_done = '1';
		drp_start <= '0';
		drp_change_mode <= '0';
		
	     -- 4. End of DRP Cycle


		wait for 2 us;
		
	    -----------------------------------------------------------------------
	    -- Test 3: Set CLKFX = 250 MHz (4000 ps) via DRP Cycle
	    -- Test 4: Set CLKFX = 400 MHz (2500 ps) via DRP Cycle
	    -----------------------------------------------------------------------

		for i in 3 to 4
		loop 

		Write ( Message, LF);
		Write ( Message, string'("-- Test "));
		Write ( Message, test_vectors(i).test_num);
		Write ( Message, string'(": *START* - Set DCM_CLKFX_OUT = "));
		Write ( Message, test_vectors(i).freq);
		Write ( Message, string'(" MHz ("));
		Write ( Message, test_vectors(i).period);
		Write ( Message, string'(") via DRP Cycle (M="));
		Write ( Message, test_vectors(i).multiplier);
		Write ( Message, string'(", D="));
		Write ( Message, test_vectors(i).divider);
		Write ( Message, string'(")")&LF);

		assert false report "" severity note;
		writeline(output, Message);

	    -- 1. Set Multiplier and Divider values from test vector
				
		drp_multiply <= conv_std_logic_vector((test_vectors(i).multiplier - 1), 8);
		drp_divide <= conv_std_logic_vector((test_vectors(i).divider - 1), 8);
		
	    -- 2. Signal DRP Controller to start a DRP cycle

		drp_start <= '1';	     
		
   	    -- 3. Wait for DCM Lock, indicating end of DRP Cycle

		WAIT UNTIL drp_done = '1';   
		drp_start <= '0';	     

		Write ( Message, LF);
		Write ( Message, string'("-- Test "));
		Write ( Message, i);
		Write ( Message, string'(": Achieved DCM LOCK ...")&LF);
		assert false report "" severity note;
		writeline(output, Message);

	    -- 4. Read simulation time for two consequent DCM_CLKFX_OUT rising edges

		WAIT until (rising_edge(dcm_clkfx_out));
		clkfx_edge_previous := NOW;
				
		WAIT until (rising_edge(dcm_clkfx_out));
		clkfx_edge_current := NOW;

	    -- 5. Compare DCM_CLKFX_OUT actual period with expected period.

		test_result(i) := period_check (i, clkfx_edge_current, clkfx_edge_previous, test_vectors(i).period);

	    -- 6. End of Test

		WAIT for 2 us;

	end loop;
		
            -----------------------------------------------------------------------
	    -- DRP Cycle Tests Completed.  Summary.  
	    -----------------------------------------------------------------------	

		Write ( Message, LF);
		Write ( Message, string'("-- DRP Cycle Tests Completed!")&LF);
		Write ( Message, string'("-- Summary:")&LF);

                for i in 1 to 4
		loop
		
		if (test_result(i)) then
			Write ( Message, string'("-- Test "));
			Write ( Message, i);
			Write ( Message, string'(": PASS")&LF);
		else
			Write ( Message, string'("-- Test "));
			Write ( Message, i);
			Write ( Message, string'(": FAIL")&LF);
		end if;

		end loop;

		assert false report "" severity note;
		writeline(output, Message);

	    -----------------------------
	    -- ** END OF SIMULATION **  
	    -----------------------------

		assert false 
		report "Simulation finished successfully (not a failure)" severity failure;

            END PROCESS;

    END behavioral;

