----------------------------------------------------------------------------------
-- Company: Xilinx, Inc.
-- Engineer: Eddie Vergara
-- 
-- Create Date:    08:38:05 03/16/2009 
-- Design Name:    DRP Demo - Controller 
-- Module Name:    drp_stmach - Behavioral 
-- Project Name:   DRP Demo
-- Target Devices: xc5vlx30
-- Tool versions:  11.1
-- Description:    The DRP Controller performs the various steps required to 
--                 complete a full DRP cycle for successfully reconfiguring 
--                 the Multiplier and Divider parameters that define the output 
--		   of the Digital Frequency Synthesizer (CLKFX_OUT).
--
--		   For more information about the DRP cycle, refer to section "Dynamic
--                 Reconfiguration Port" in the Virtex-5 Configuration Guide (UG191).
--
--		   http://www.xilinx.com/support/documentation/virtex-5.htm
--
----------------------------------------------------------------------------------


library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity drp_stmach is
    Port ( clk : in  STD_LOGIC;
           start : in  STD_LOGIC;
           reset : in  STD_LOGIC;
	   
           new_freq_mode : in STD_LOGIC;
           
	   drp_ready : in  STD_LOGIC;
           drp_en : out  STD_LOGIC;
           drp_we : out  STD_LOGIC;
           drp_addr : out  STD_LOGIC_VECTOR (6 downto 0);
	   drp_sel : out STD_LOGIC;
	   
	   dcm_locked : in  STD_LOGIC;
           dcm_reset : out  STD_LOGIC;
	   
	   drp_done : out  STD_LOGIC);
end drp_stmach;

architecture Behavioral of drp_stmach is

   type state_type is (
   st0_stat_reset, st1_stat_idle, 
   st2_dcm_clear_regs, st3_dcm_reset, 
   st4_DRP_read_start, st5_DRP_read_finish, 
   st6_DRP_write_start, st7_DRP_write_finish,
   st8_DCM_clear_regs, st9_DCM_start, st10_DCM_locked);

   signal state, next_state : state_type; 
   signal drp_en_i, drp_we_i, dcm_reset_i : std_logic;
   signal drp_done_i : STD_LOGIC;  
   signal drp_addr_i : std_logic_vector (6 downto 0);
   signal drp_sel_i : std_logic;
    
   
begin

   SYNC_PROC: process (clk)
   begin
      if (clk'event and clk = '1') then
         if (reset = '1') then
            state <= st0_stat_reset;
	    drp_en <= '0';
	    drp_we <= '0';
	    drp_addr <= "1010000";
	    drp_sel <= '0';
	    dcm_reset <= '1';
	    drp_done <= '0';
	 else
            state <= next_state;
	    drp_en <= drp_en_i;
	    drp_we <= drp_we_i;
	    drp_addr <= drp_addr_i;
	    drp_sel <= drp_sel_i;
	    dcm_reset <= dcm_reset_i;
	    drp_done <= drp_done_i;
         end if;        
      end if;
   end process;
 
   NEXT_STATE_DECODE: process (state, reset, start, drp_ready, dcm_locked)
   begin
      next_state <= state; 
      
      drp_en_i <= '0';
      drp_we_i <= '0';
      drp_addr_i <= "1010000";
      drp_sel_i <= '0';
      dcm_reset_i <= '0';
      drp_done_i <= '0';
            
      case (state) is
	    
         when st0_stat_reset =>
            if reset = '1' then				
               next_state <= st0_stat_reset;		-- Reset
	       dcm_reset_i <= '1';
	    else
       	       next_state <= st1_stat_idle;		-- Idle
	       drp_done_i <= '1';
            end if;
	   
         when st1_stat_idle =>
            if start = '1' then				-- If DRP Start signal asserted, 
	    next_state <= st2_dcm_clear_regs;		-- clear DCM Status registers
	       drp_addr_i <= "0000000";
               drp_en_i <= '1';
	    else
	       next_state <= st1_stat_idle;		-- otherwise remain in idle 
	       drp_done_i <= '1';
            end if;

	 when st2_dcm_clear_regs =>
	    if drp_ready = '1' then			-- if DRP Ready signal asserted,
		next_state <= st3_dcm_reset;		-- hold DCM in reset
	         
	    else
	       next_state <= st2_dcm_clear_regs;	-- otherwise wait for DRP ready signal 
	    	       
	    end if;
	    
	    dcm_reset_i <= '1';
	    drp_en_i <= '0';
	 
	 when st3_dcm_reset =>
	    next_state <= st4_DRP_read_start;		-- Perform DRP Ready Cycle 
	    dcm_reset_i <= '1';
	    drp_en_i <= '1';

	       if(new_freq_mode = '1') then		-- If New Frequency Mode,
	          drp_sel_i <= '1';			
		  drp_addr_i <= "1000001";		-- Read from DRP Address 0x51 
	       else
		  drp_sel_i <= '0';
		  drp_addr_i <= "1010000";
	       end if;

	 when st4_DRP_read_start =>
	    
	    if drp_ready = '1' then			-- if DRP Ready signal asserted, 
              
	       next_state <= st5_DRP_read_finish;	-- prepare for DRP Write Cycle
	         
	    else
	       next_state <= st4_DRP_read_start;	-- otherwise wait for DRP ready signal 
	       
	       
	    end if;
	     
	    drp_en_i <= '0';   
	    dcm_reset_i <= '1';

	 when st5_DRP_read_finish =>
	    next_state <= st6_DRP_write_start;		-- Perform DRP Write Cycle 
	    
	    dcm_reset_i <= '1';
	    drp_we_i <= '1';
	    drp_en_i <= '1';
	    
	       if(new_freq_mode = '1') then		-- If New Frequency Mode,
	          drp_sel_i <= '1';
		  drp_addr_i <= "1000001";		-- Write to DRP Address 0x51 
	       else
		  drp_sel_i <= '0';
		  drp_addr_i <= "1010000";
	       end if;
    
	 when st6_DRP_write_start =>
	    
	    if drp_ready = '1' then			-- if DRP Ready signal asserted,
 
	       next_state <= st7_DRP_write_finish;	-- prepare for DRP Cycle Finish	
	      
	    else
	       next_state <= st6_DRP_write_start;	-- otherwise wait for DRP ready signal 
	          
	    end if;
	    
	    dcm_reset_i <= '1';
	    drp_we_i <= '0';
	    drp_en_i <= '0';
	    
----------------------------------------------
-- Wait for DRP Write Cycle
----------------------------------------------	

	 when st7_DRP_write_finish =>
	    next_state <= st8_dcm_clear_regs;		-- Clear DCM Status registers
	    
	    dcm_reset_i <= '0';
	    drp_we_i <= '0';
	    drp_addr_i <= "0000000";
            drp_en_i <= '1';
	    
	   
	 when st8_dcm_clear_regs =>
	    next_state <= st9_DCM_start;		-- De-assert DCM Reset
	    
	    dcm_reset_i <= '0';
	    drp_en_i <= '0';
	    
	    
	 when st9_DCM_start =>
	    if dcm_locked = '1' then			-- if DCM LOCKED signal is asserted, 
             
	       next_state <= st10_DCM_locked;		-- Signal End of DRP Cycle
	       
	       drp_done_i <= '1';
	       
	    else
	       next_state <= st9_DCM_start;		-- otherwise wait for DCM LOCKED signal 
	       
			 drp_done_i <= '0';
       end if;
	    
	 when st10_DCM_locked =>
            next_state <= st1_stat_idle;		-- End of DRP Cycle.  Return to Idle.
	    
	    drp_done_i <= '1';
	    
	 when others =>					-- Default State: Idle.
	    next_state <= st1_stat_idle;
	    
	    drp_done_i <= '1';
	    	    
      end case;      
   end process;
   
end Behavioral;























