#ifdef __GNUC__
/* $Id: xvectors.S,v 1.1.2.1 2011/05/17 04:37:54 sadanan Exp $ */
//$xilinx_legal_disclaimer
/****************************************************************************/
/**
*
* @file xvectors.S
*
* This file contains low-level functions for the PowerPC exception handler.
*
* <pre>
* MODIFICATION HISTORY:
*
* Ver   Who  Date     Changes
* ----- ---- -------- -----------------------------------------------
* 1.00a ch   06/18/02 First release
* </pre>
*
*****************************************************************************/

/*-----------------------------------------------------------------------------
//
// Documentation:
//
//    This is the structure of the PPC 440 interrupt stack frame that is
//    pushed onto the stack during an exception.
//
//    This interrupt frame only saves the User Model register sets that
//    could be used/generated by compiler-generated code. Any usage of
//    registers in an ISR beyond this set must be saved/restored by the
//    ISR routine.
//
//    In general, this interrupt frame conforms to a standard PPC EABI
//    Stack Call Frame. The entire frame is 160 bytes (40 words) in size,
//    which is an exact multiple of 8 bytes as required by the PPC EABI.
//    And by using the standard EABI frame header, debuggers should still
//    be able to display the Stack Call Chain even for break points in an
//    exception handler function.
//
//    However, there are a few exceptions to EABI conventions :
//
//        1. The frame saves ALL registers instead of just registers that
//           will be modified. Exceptions can occur at any point instead
//           of just at function call branch points. Hence, saving the entire
//           context means an exception will never disturb the interrupted
//           code's register usage.
//
//        2. The current LR register contents is NOT saved in the previous
//           Stack Frame as in standard EABI convention. Rather, it is
//           saved in the local frame. This is done to avoid corrupting the
//           previous Stack Frame's LR field, when the exception occurs
//           during an interrupted function's entry prologue process.
//
//        3. Unlike standard EABI frames, the interrupted PC location, and
//           MSR state is also saved to the frame. This may seem unnecessary
//           since registers SRR0, SRR1, CSRR0, CSRR1, MCSRR0, MCSRR1 in the 
//           440 are used exclusively to save the PC and MSR during 
//           exceptions. However, always saving the PC and MSR prepares this
//           code for use in a multi-threaded environment.
//
//        4. The USPRG0 register is also saved even though compilers will
//           never generate code to use it. Once again, this prepares the
//           way for use in a multi-threaded environment. This register
//           would be used to contain a pointer to the TCB (Task Control
//           Block) for the current task that is being interrupted.
//           Subsequent system code could then switch the memory map/context
//           from the TCB; such as the PID register.
//
//    +------------+
//    |     r31    | +156
//    |      .     |
//    |      .     |
//    |      .     |
//    |     r3     | +44
//    +------------+
//    |     r2     | +40
//    +------------+
//    |     r0     | +36
//    +------------+
//    |   USPRG0   | +32
//    +------------+
//    |     CR     | +28
//    +------------+
//    |     XER    | +24
//    +------------+
//    |     CTR    | +20
//    +------------+
//    |     LR     | +16
//    +------------+
//    |     PC     | +12
//    +------------+
//    |     MSR    | +8
//    +------------+
//    |   Next LR  | +4
//    +------------+
//    | Back Chain | +0
//    +------------+
//
//---------------------------------------------------------------------------*/
	
#endif

/*-----------------------------------------------------------------------------
//
// How the following routines work:
// The interrupt initialization routine, XExc_Init(), sets up the IVPR and
// IVOR0 - IVOR15 registers. The IVPR is mapped to a vector, _vectorbase,
// which contains code to branch to the appropriate handler.
// Flow of execution:
//   Interrupt/Exception asserts
//   PC is set to corresponding entry in _vectorbase, e.g. IVOR10
//   Either a critical, non-critical or machine check handler is called,
//   in which, the interrupt prologue, "full" handler, and then the
//   interrupt epilogue is called.
//
//---------------------------------------------------------------------------*/
			
#include "xreg440.h"

	.file "xvectors.S"
	.section ".vectors","ax"
	
#ifdef __DCC__
#define // ;
#define MACRO0(name)name:	.macro
#define MACRO1(name, param1)name:	.macro	param1
#define CONCAT(left, right) left##&&##right	
#define CONCAT3(left, parammiddle, right) left##&&##parammiddle##right	
#define PARAM(param) param	
#else	
#include <ppc-asm.h>
#define r1 1
#define r2 2
#define MACRO0(name) .macro	name
#define MACRO1(name, param1) .macro	name	param1
#define CONCAT(left,right) left##right		
#define PARAM(param) param		
#endif	
	
	//---------------------------------------------------------------------
	//
	// Define interrupt frame structure offsets.
	//
	//---------------------------------------------------------------------

	.set	BChainField, 0
	.set	NextLRField, BChainField + 4
	.set	MSRField,    NextLRField + 4
	.set	PCField,     MSRField    + 4
	.set	LRField,     PCField     + 4
	.set	CTRField,    LRField     + 4
	.set	XERField,    CTRField    + 4
	.set	CRField,     XERField    + 4
	.set	USPRG0Field, CRField     + 4
	.set	r0Field,     USPRG0Field + 4
	.set	r2Field,     r0Field     + 4
	.set	r3r31Field,  r2Field     + 4
	.set	IFrameSize,  r3r31Field  + ( ( 31 - 3 ) + 1 ) * 4
	
	
	//---------------------------------------------------------------------
	//
	// Macro for the interrupt frame prologue.
	//
	//---------------------------------------------------------------------
		
	MACRO1(int_prologue, base)
	
	.globl	CONCAT(_vector,\base)
 	.org	_vectorbase + CONCAT(0x,\base)    
#ifdef __DCC__
CONCAT3(_vector, base, :)
#else
_vector\base:                   
#endif        
	stwu	XREG_GPR1,-IFrameSize(XREG_GPR1)  // Save Back chain and move SP
	
	// Save r0 and r2.
	stw	XREG_GPR0, r0Field(XREG_GPR1)
	stw	XREG_GPR2, r2Field(XREG_GPR1)

	stmw	XREG_GPR3, r3r31Field(XREG_GPR1)  // Save the remaining GPR registers r3-r31
	
	// Save current Link register
	mflr	XREG_GPR0
	stw	XREG_GPR0, LRField(XREG_GPR1)
	
	// Save Counter register
	mfctr	XREG_GPR0
	stw	XREG_GPR0,CTRField(XREG_GPR1)
	
	// Save Fixed Point Exception register
	mfxer	XREG_GPR0
	stw	XREG_GPR0,XERField(XREG_GPR1)
	
	// Save Condition register
	mfcr	XREG_GPR0
	stw	XREG_GPR0,CRField(XREG_GPR1)
	
	// Save USPRG0 register
	mfspr	XREG_GPR0,XREG_SPR_USPRG0
	stw	XREG_GPR0,USPRG0Field(XREG_GPR1)
	
	.endm
	
	
	//---------------------------------------------------------------------
	//
	// Macro for the interrupt frame epilogue.
	//
	//---------------------------------------------------------------------
	
	MACRO0(int_epilogue)
	
	// Save USPRG0 register
	lwz	XREG_GPR0,USPRG0Field(XREG_GPR1)
	mtspr	XREG_SPR_USPRG0,XREG_GPR0
	
	// Restore Condition register
	lwz	XREG_GPR0,CRField(XREG_GPR1)
	mtcr	XREG_GPR0
	
	// Restore Fixed Point Exception register
	lwz	XREG_GPR0,XERField(XREG_GPR1)
	mtxer	XREG_GPR0
	
	// Restore Counter register
	lwz	XREG_GPR0,CTRField(XREG_GPR1)
	mtctr	XREG_GPR0
	
	// Restore Link register
	lwz	XREG_GPR0,LRField(XREG_GPR1)
	mtlr	XREG_GPR0
	
	// Restore remaining GPR registers.
	lmw	XREG_GPR3,r3r31Field(XREG_GPR1)
	
	// Restore r0 and r2.
	lwz	XREG_GPR0,r0Field(XREG_GPR1)
	lwz	XREG_GPR2,r2Field(XREG_GPR1)
	
	// Remove frame from stack
	addi	XREG_GPR1,XREG_GPR1,IFrameSize
	
	.endm
	
	
	//---------------------------------------------------------------------
	//
	// Macro for vector handler calling.
	//
	//---------------------------------------------------------------------
	
	MACRO0(call_vector_handler)
	
  	// Get address of the vector table in r9.
	lis	XREG_GPR9,XExc_VectorTable@ha
	la	XREG_GPR9,XExc_VectorTable@l(XREG_GPR9)
	
	// Multiply vectorNumber (from r3) by 16 into r0.
	mr	XREG_GPR11,XREG_GPR3
	slwi	XREG_GPR0,XREG_GPR11,4
  	
  	// Get the Nth vector element at r9 + r0 into r9.
	add	XREG_GPR9,XREG_GPR9,XREG_GPR0
  	
  	// Get the vector address into r11.
	lwz	XREG_GPR11,0(XREG_GPR9)
  	
  	// Get the user data value into r4.
	lwz	XREG_GPR3,4(XREG_GPR9)
  	
  	// Get the read only small data area pointer into r2.
	lwz	XREG_GPR2,8(XREG_GPR9)
  	
  	// Get the read/write small data area pointer into r13.
	lwz	XREG_GPR13,12(XREG_GPR9)

	// NOT! Pass exception ordinal parameter in r3.
	// Pass user data value parameter in r4.
  	
  	// Move the vector address to the LR and call it.
	mtlr	XREG_GPR11
	blrl
	
	.endm

	//---------------------------------------------------------------------
	//
	// Macro for machine check interrupts vector code.
	//
	//---------------------------------------------------------------------
	
	MACRO1(machine_check, base)

	mfmsr    XREG_GPR0                    // read MSR
	ori	 XREG_GPR0,XREG_GPR0,0x10     // reset MSR[IS,DS] bits
	mtmsr	 XREG_GPR0
	mfspr	 XREG_GPR0,XREG_SPR_SPRG0_SU  // restore r0 for prologue 
		
	int_prologue PARAM(\base)
		
	// Save MSR register from MCSRR1.
	mfmcsrr1 XREG_GPR0
	stw	 XREG_GPR0,MSRField(XREG_GPR1)
	
	// Save current PC location from MCSRR0.
	mfmcsrr0  XREG_GPR0
	stw	 XREG_GPR0,PCField(XREG_GPR1)
	
	// move index from SPRG1 to r3
	mfspr   XREG_GPR3,XREG_SPR_SPRG1_SU
	
	// Call the vector handler for this exception.
	call_vector_handler
			
	// Restore MSR register to MCSRR1.
	lwz	 XREG_GPR0,MSRField(XREG_GPR1)
#ifdef PPC440_RESET_WE_ON_RFI
        rlwinm   XREG_GPR0,XREG_GPR0,0,(XREG_MSR_WAIT_STATE_ENABLE_BIT_POS + 1), (XREG_MSR_WAIT_STATE_ENABLE_BIT_POS - 1)    
#endif
        mtmcsrr1 XREG_GPR0
	
	// Restore current PC location to MCSRR0.
	lwz	 XREG_GPR0,PCField(XREG_GPR1)
	mtmcsrr0 XREG_GPR0
	
	int_epilogue
	
	// Return to the interrupted code
	rfmci
	
	.endm
		
	
	//---------------------------------------------------------------------
	//
	// Macro for critical interrupts vector code.
	//
	//---------------------------------------------------------------------
	
	MACRO1(critical_interrupt, base)

	mfmsr   XREG_GPR0                    // read in MSR
	ori	XREG_GPR0,XREG_GPR0,0x10     // reset MSR[IS,DS] bits
	mtmsr	XREG_GPR0
	mfspr	XREG_GPR0,XREG_SPR_SPRG2_SU  // restore r0 for prologue 
		
	int_prologue PARAM(\base)
		
	// Save MSR register from CSRR1.
	mfcsrr1	XREG_GPR0
	stw	XREG_GPR0,MSRField(XREG_GPR1)
	
	// Save current PC location from CSRR0.
	mfcsrr0	XREG_GPR0
	stw	XREG_GPR0,PCField(XREG_GPR1)

	// move index from SPRG3 to r3
	mfspr   XREG_GPR3,XREG_SPR_SPRG3_SU
		
	// Call the vector handler for this exception.
	call_vector_handler
			
	// Restore MSR register to CSRR1.
	lwz	XREG_GPR0,MSRField(XREG_GPR1)
#ifdef PPC440_RESET_WE_ON_RFI
        rlwinm  XREG_GPR0,XREG_GPR0,0,(XREG_MSR_WAIT_STATE_ENABLE_BIT_POS + 1), (XREG_MSR_WAIT_STATE_ENABLE_BIT_POS - 1)    
#endif
   	mtcsrr1	XREG_GPR0
	
	// Restore current PC location to CSRR0.
	lwz	XREG_GPR0,PCField(XREG_GPR1)
	mtcsrr0	XREG_GPR0
	
	int_epilogue
	
	// Return to the interrupted code
	rfci
	
	.endm
	
	
	//---------------------------------------------------------------------
	//
	// Macro for non-critical interrupts vector code.
	//
	//---------------------------------------------------------------------
	
	MACRO1(non_critical_interrupt, base)

	mfmsr   XREG_GPR0                    // read in MSR
	ori	XREG_GPR0,XREG_GPR0,0x10     // reset MSR[IS,DS] bits
	mtmsr	XREG_GPR0
	mfspr	XREG_GPR0,XREG_SPR_SPRG4_U   // restore r0 for prologue 
	
	int_prologue PARAM(\base)
	
	// Save MSR register from SRR1.
	mfsrr1	XREG_GPR0
	stw	XREG_GPR0,MSRField(XREG_GPR1)
	
	// Save current PC location from SRR0.
	mfsrr0	XREG_GPR0
	stw	XREG_GPR0,PCField(XREG_GPR1)

	// move index from SPRG5 to r3
	mfspr   XREG_GPR3,XREG_SPR_SPRG5_U
		
	// Call the vector handler for this exception.
	call_vector_handler
	
	// Restore MSR register to SRR1.
	lwz	XREG_GPR0,MSRField(XREG_GPR1)
#ifdef PPC440_RESET_WE_ON_RFI
        rlwinm  XREG_GPR0,XREG_GPR0,0,(XREG_MSR_WAIT_STATE_ENABLE_BIT_POS + 1), (XREG_MSR_WAIT_STATE_ENABLE_BIT_POS - 1)    
#endif
        mtsrr1	XREG_GPR0
	
	// Restore current PC location to SRR0.
	lwz	XREG_GPR0,PCField(XREG_GPR1)
	mtsrr0	XREG_GPR0
	
	int_epilogue
	
	// Return to the interrupted code
	rfi
	
	.endm
	
	
	//---------------------------------------------------------------------
	//
	// Vector table.
	//
	// Define the vector table for all exceptions. The entry points
	// listed are used to set an index variable and then branch
	// to the appropriate pre-handler setup.  The IVOR registers
	// are set to their respective entry points when XExc_Init is
	// invoked in xexception.
	// The two pre-handlers are inlined at the point of the label.
	//
	// NOTE:
	//	In the issue of memory space, the choice of handler speed
	// over memory usage it was chosen to have three handlers to keep
	// performance close to previous versions for previous PPC procesors.
	// It is possible to combine the handlers into one at the cost of
	// adding more instructions per handler and possibly hindering 
	// performance.
	//
	// NOTE:
	//	The resetting of the MSR[IS,DS] bits is done to comply
	// with changes made in Xcache that are backwards compatible with
	// previous PPC processors.  If the backwards compatible versions
	// of files are not being used, then the code that resets the
	// MSR[IS,DS] bits can be removed.
	//
	// SPRG0-5 are used for saving GPR0 and the index of the handler in
	// the exception handler table.
	//   0,1: Machine Check interrupts
	//   2,3: Critical interrupts
	//   4,5: Non-critical interrupts (Use 0x104,0x105 to read and
	//                                     0x114,0x115 to write)
	//
	// If in the future this code is used for a different PPC processor,
	// this vector table may need to be changed.
	//
	//---------------------------------------------------------------------
	
	.globl	_vectorbase
	.align  4               // 16 byte alignment to ensure the following IVORs are aligned correctly
_vectorbase:
	
	// "Vector 0x00", Critical interrupt.
	IVOR0:  mtspr XREG_SPR_SPRG2_SU,XREG_GPR0  // move R0 to SPRG2
		li    XREG_GPR0,0
		mtspr XREG_SPR_SPRG3_SU,XREG_GPR0  // move index param to SPRG3
		b     critical
	
	// "Vector 0x10", Machine Check interrupt.
	IVOR1:  mtspr XREG_SPR_SPRG0_SU,XREG_GPR0  // move R0 to SPRG0
		li    XREG_GPR0,1
		mtspr XREG_SPR_SPRG1_SU,XREG_GPR0  // move index param to SPRG1
		b     machine
	
	// "Vector 0x20", Data Storage interrupt.
	IVOR2:  mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,2
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	// "Vector 0x30", Instruction Storage interrupt.
	IVOR3:  mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,3
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	// "Vector 0x40", External interrupt.
	IVOR4:  mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,4
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	// "Vector 0x50", Alignment interrupt.
	IVOR5:  mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,5
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	// "Vector 0x60", Program interrupt.
	IVOR6:  mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,6
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	//" Vector 0x70", FPU Unavailable interrupt.
	IVOR7:  mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,7
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	// "Vector 0x80", System Call interrupt.
	IVOR8:  mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,8
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	// "Vector 0x90", APU Available interrupt.
	IVOR9:  mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,9
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	// "Vector 0xA0", DEC interrupt.
	IVOR10: mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,10
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
		
	// "Vector 0xB0", FIT interrupt.
	IVOR11: mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,11
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	// "Vector 0xC0", Watchdog Timer interrupt.
	IVOR12: mtspr XREG_SPR_SPRG2_SU,XREG_GPR0  // move R0 to SPRG2
		li    XREG_GPR0,12
		mtspr XREG_SPR_SPRG3_SU,XREG_GPR0  // move index param to SPRG3
		b     critical
	
	// "Vector 0xD0", Data TLB Miss interrupt.
	IVOR13: mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,13
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG5
		b     noncritical
	
	// "Vector 0xE0", Instruction TLB Miss interrupt.
	IVOR14: mtspr XREG_SPR_SPRG4_SU,XREG_GPR0  // move R0 to SPRG4
		li    XREG_GPR0,14
		mtspr XREG_SPR_SPRG5_SU,XREG_GPR0  // move index param to SPRG3
		b     noncritical

	// "Vector 0xF0", Debug interrupt.
	IVOR15: mtspr XREG_SPR_SPRG2_SU,XREG_GPR0  // move R0 to SPRG2
		li    XREG_GPR0,15
		mtspr XREG_SPR_SPRG3_SU,XREG_GPR0  // move index param to SPRG3
		b     critical

/* Calculate offsets from vector table to each handler so as to place the int_prologue macro */
/* 0x110=(256+16)d = _vectortable + msr set instr */
noncritical:	non_critical_interrupt 110
/* 0x1E4=(256+212+16)d = _vectortable + noncritical + 4 msr[is,ds] set instructions */
critical:	critical_interrupt 1E4
/* 0x2B8=(256+212+212+16)d= _vectortable + noncritical + critical + 4 msr[is,ds] set instructions */
machine:	machine_check 2B8
