#!/usr/bin/env python3
# collect_dtypes.py
import argparse
import ast
import json
from pathlib import Path

def find_json_for_folder(folder: Path) -> Path | None:
    """
    Prefer <folder>/<folder.name>.json.
    If not present, use the only *.json in the folder.
    """
    exact = folder / f"{folder.name}.json"
    if exact.exists():
        return exact
    jsons = list(folder.glob("*.json"))
    if len(jsons) == 1:
        return jsons[0]
    return None

def extract_out_dtype(data: dict) -> str | None:
    """
    Try layer_info.out_datatype first.
    Fallback: parse layer_info.outputs (a Python-literal string) and look for dtype on the act output.
    """
    li = data.get("layer_info", {})
    # Primary key(s)
    for k in ("out_ofm_datatype", "in_datatype", "in_dtype", "input_datatype", "in_ifm_datatype"): 
        v = li.get(k) 
        x = li.get("attributes") 
        if(x["disable_q"][0]==1): 
            return "bf16" 

        if isinstance(v, str) and v: 
            return v 

    # Fallback: outputs as a Python literal string
    outputs = li.get("outputs")
    if isinstance(outputs, str) and outputs.strip():
        try:
            lst = ast.literal_eval(outputs)  # safe parse of Python literal
            if isinstance(lst, list):
                for item in lst:
                    if isinstance(item, dict) and item.get("type") == "act":
                        dt = item.get("dtype")
                        if isinstance(dt, str) and dt:
                            return dt
        except Exception:
            pass

    return None

def main():
    ap = argparse.ArgumentParser(description="Collect out_datatype from per-folder JSON files.")
    ap.add_argument("--root", default=".", help="Root directory containing subfolders (default: .)")
    ap.add_argument("--out", default="dtype_map.json", help="Output JSON path (default: dtype_map.json)")
    ap.add_argument("--quiet", action="store_true", help="Suppress warnings")
    args = ap.parse_args()

    root = Path(args.root).resolve()
    if not root.is_dir():
        raise SystemExit(f"ERROR: {root} is not a directory")

    result: dict[str, str] = {}
    missing = []

    for sub in sorted([p for p in root.iterdir() if p.is_dir()], key=lambda p: p.name):
        jf = find_json_for_folder(sub)
        if not jf:
            if not args.quiet:
                print(f"WARNING: no JSON found for folder {sub.name}")
            continue

        try:
            data = json.loads(jf.read_text(encoding="utf-8"))
        except Exception as e:
            if not args.quiet:
                print(f"WARNING: failed to parse {jf}: {e}")
            continue

        dtype = extract_out_dtype(data)
        if dtype:
            result[sub.name] = dtype
        else:
            missing.append(sub.name)
            if not args.quiet:
                print(f"WARNING: no out_datatype found in {jf}")

    Path(args.out).write_text(json.dumps(result, indent=2) + "\n", encoding="utf-8")

    print(f"Wrote {args.out} with {len(result)} entries.")
    if missing and not args.quiet:
        print(f"{len(missing)} folders had no dtype (see warnings).")

if __name__ == "__main__":
    main()
