import json
import os
import sys

# Set this flag to True for new PM ID assignment flow
NEW_PM_ASSIGNMENT_FLOW = False


def collect_pm_ids(input_dir, output_dir):
    result = []

    # Load IR JSON for new flow
    ir_data = None

    if NEW_PM_ASSIGNMENT_FLOW:
        # Find the IR JSON file (there should be one in the input directory (WAIC Outputs))
        import glob

        ir_files = glob.glob(os.path.join(input_dir, "*_mod_nhwc_fused_IR.json"))
        if not ir_files:
            print(
                f"ERROR: No IR JSON file (*_mod_nhwc_fused_IR.json) found in {input_dir}"
            )
            sys.exit(1)

        ir_json_path = ir_files[0]
        with open(ir_json_path, "r") as f:
            ir_data = json.load(f)
        print(f"Loaded IR JSON: {ir_json_path}")

    for subfolder in os.listdir(input_dir):
        subfolder_path = os.path.join(input_dir, subfolder)
        if os.path.isdir(subfolder_path):
            json_file = os.path.join(subfolder_path, f"{subfolder}.json")
            if os.path.isfile(json_file):
                try:
                    with open(json_file, "r") as f:
                        data = json.load(f)

                    if NEW_PM_ASSIGNMENT_FLOW:
                        # New flow: Extract nodenames from folder JSON, get pm_id from IR JSON
                        nodenames = data.get("layer_info", {}).get("nodenames", [])

                        if isinstance(nodenames, list) and nodenames and ir_data:
                            # Get the first nodename
                            nodename = nodenames[0]

                            # Look up pm_id in IR JSON
                            if nodename in ir_data:
                                pm_id = (
                                    ir_data[nodename]
                                    .get("attributes", {})
                                    .get("pm_id", [])
                                )
                                if isinstance(pm_id, list) and pm_id:
                                    result.append(
                                        {
                                            "folder": subfolder,
                                            "nodename": nodename,
                                            "pm_id": pm_id[0],
                                        }
                                    )
                                else:
                                    # No pm_id in IR for this nodename
                                    result.append(
                                        {
                                            "folder": subfolder,
                                            "nodename": nodename,
                                            "pm_id": 0,
                                        }
                                    )
                            else:
                                # Nodename not found in IR JSON
                                print(
                                    f"Warning: nodename '{nodename}' not found in IR JSON"
                                )
                                result.append(
                                    {
                                        "folder": subfolder,
                                        "nodename": nodename,
                                        "pm_id": 0,
                                    }
                                )
                        else:
                            # No nodenames found in folder JSON
                            result.append(
                                {"folder": subfolder, "nodename": None, "pm_id": 0}
                            )
                    else:
                        # Old flow: Navigate safely to layer_info → attributes → pm_id
                        pm_id = (
                            data.get("layer_info", {})
                            .get("attributes", {})
                            .get("pm_id", [])
                        )

                        if isinstance(pm_id, list) and pm_id:
                            # Take the first pm_id if available
                            result.append({"folder": subfolder, "pm_id": pm_id[0]})
                        else:
                            # JSON exists but no pm_id found — assign default 0
                            result.append({"folder": subfolder, "pm_id": 0})

                except Exception as e:
                    print(f"Error reading {json_file}: {e}")
                    result.append({"folder": subfolder, "pm_id": 0})
            else:
                # JSON file is missing — assign default pm_id 0
                result.append({"folder": subfolder, "pm_id": 0})

    os.makedirs(output_dir, exist_ok=True)
    output_file = os.path.join(output_dir, "pm_id_list.json")
    with open(output_file, "w") as f:
        json.dump(result, f, indent=2)

    print(f"Saved {len(result)} entries to {output_file}")


if __name__ == "__main__":
    script_dir = os.path.dirname(os.path.abspath(__file__))
    if len(sys.argv) == 3:
        input_dir = os.path.abspath(sys.argv[1])
        output_dir = os.path.abspath(sys.argv[2])
    else:
        print(
            "No input/output directories provided — using script location as default."
        )
        input_dir = output_dir = script_dir

    collect_pm_ids(input_dir, output_dir)
