from datetime import datetime
import subprocess
import os
import shutil
import tarfile
import tempfile
import json
import glob
import sys
from dotenv import load_dotenv
import platform


def _is_windows_local():
    # This script itself is running on a Windows machine (the DUT)
    return platform.system().lower().startswith("win")


load_dotenv("/actions-runners/.env")

if platform.system().lower().startswith("win"):
    import importlib, subprocess, sys

    def ensure_pkg(pkg):
        try:
            importlib.import_module(pkg)
        except ImportError:
            print(f"[SETUP] Installing missing package: {pkg}")
            subprocess.run(
                [sys.executable, "-m", "pip", "install", pkg, "--quiet"], check=False
            )

    for pkg in ["paramiko", "openpyxl", "prettytable", "pandas"]:
        ensure_pkg(pkg)


def HW_test(
    output: str,
    HW_req: str,
    xclbin: int,
    xclbin_path: str,
    overlay: str,
    use_bsub: bool,
    host: str,
    perf_testing=False,
    golden_io: list[str] | None = None,
    rename=False,
    profile_perf=False,
    rel_err_pc=False,
    disable_fast_pm=False,
    dtype=False,
):
    """
    Run test on hardware.
    output -- absolute path to output directory
    HW_req -- absolute path to HW_requirements directory of repository
    xclbin -- 1: generate xclbin, 0: expect xclbin in xclbin_path
    xclbin_path -- absolute path to directory containing out.xclbin
    overlay -- overlay as string, e.g., '4x4' or '8x4'
    use_bsub -- True: launch build process via bsub on LSF cluster,
                            False: run build process as subprocess on the same machine
    """

    from dataflow.xclbin.auto_xclbin import gen_xclbin

    def read_record_timer_ts(file_path):
        r = []
        try:
            with open(file_path, "r") as file:
                data = json.load(file)
            with open(file_path.replace(".json", "_result.txt"), "w") as mf:
                for x in data["record_timer_ts"]:
                    if x["id"] == 100:
                        xstart = x["cycle"]
                    elif x["id"] == 101:
                        mf.write(f'{x["cycle"] - xstart}\n')
                        r.append(x["cycle"] - xstart)
            temp = sorted(r[-800:])
            return round(((temp[399] + temp[400]) / 2) / 1800, 2)
        except FileNotFoundError:
            print(f"File {file_path} not found.")
            return None
        except json.JSONDecodeError:
            print(f"Error decoding JSON from {file_path}.")
            return None

    def rename_local_files(src, search_file="ctrl_meta.json", new_name="patch.json"):
        try:
            if not os.path.exists(src):
                print(f"Source path '{src}' does not exist.")
                return
            for root, dirs, files in os.walk(src):
                for file in files:
                    if file == search_file:
                        src_file = os.path.join(root, file)
                        new_file = os.path.join(root, new_name)
                        os.rename(src_file, new_file)
                        print(f"Renamed '{src_file}' to '{new_file}'")
        except Exception as e:
            print(f"An error occurred: {e}")

    def copy_file_with_smbclient(src, dst, host, username, password):
        try:
            if not os.path.exists(src):
                print(f"Source path '{src}' does not exist.")
                return
            dst_path = os.path.join(dst, os.path.basename(src))
            smbclient_command = f'smbclient //{host}/Users -U {username}%{password} -c \'put "{src}" "{dst_path}"; exit\''
            subprocess.run(smbclient_command, shell=True, check=True)
            print(f"Copied '{src}' to '{dst}'")
        except Exception as e:
            print(f"An error occurred: {e}")

    def create_dir_with_smbclient(dst, host, username, password):
        try:
            smbclient_command = f"smbclient //{host}/Users -U {username}%{password} -c 'mkdir \"{dst}\"; exit'"
            subprocess.run(smbclient_command, shell=True, check=True)
            print(f"Created directory '{dst}'")
        except Exception as e:
            print(f"An error occurred: {e}")

    def copy_files_with_smbclient(src, dst, host, username, password, file_list=None):
        try:
            if not os.path.exists(src):
                print(f"Source path '{src}' does not exist.")
                return
            dst = os.path.join(dst, timestamp)
            smbclient_command = f"smbclient //{host}/Users -U {username}%{password} -c 'mkdir \"{dst}\"; exit'"
            subprocess.run(smbclient_command, shell=True, check=True)
            print(f"Created directory '{dst}'")
            for root, dirs, files in os.walk(src):
                for dir in dirs:
                    src_dir = os.path.join(root, dir)
                    rel_path = os.path.relpath(src_dir, src)
                    dst_dir = os.path.join(dst, rel_path).replace("\\", "/")
                    subfolder_files = [
                        f
                        for f in os.listdir(src_dir)
                        if os.path.isfile(os.path.join(src_dir, f))
                    ]
                    files_to_copy = (
                        [f for f in subfolder_files if f in file_list]
                        if file_list
                        else subfolder_files
                    )
                    if files_to_copy:
                        smbclient_command = f"smbclient //{host}/Users -U {username}%{password} -c 'mkdir \"{dst_dir}\"; exit'"
                        subprocess.run(smbclient_command, shell=True, check=True)
                        print(f"Created directory '{dst_dir}'")
                        for file in files_to_copy:
                            src_file = os.path.join(src_dir, file)
                            dst_file = os.path.join(dst_dir, file).replace("\\", "/")
                            smbclient_command = f'smbclient //{host}/Users -U {username}%{password} -c \'put "{src_file}" "{dst_file}"; exit\''
                            subprocess.run(smbclient_command, shell=True, check=True)
                            print(f"Copied '{src_file}' to '{dst_file}'")
                for file in files:
                    if not file_list or file in file_list:
                        src_file = os.path.join(root, file)
                        rel_path = os.path.relpath(src_file, src)
                        dst_file = os.path.join(dst, rel_path).replace("\\", "/")
                        smbclient_command = f'smbclient //{host}/Users -U {username}%{password} -c \'put "{src_file}" "{dst_file}"; exit\''
                        subprocess.run(smbclient_command, shell=True, check=True)
                        print(f"Copied '{src_file}' to '{dst_file}'")
        except Exception as e:
            print(f"An error occurred: {e}")

    def copy_and_untar_allfolders(
        base_dir,
        remote_base,
        host,
        username,
        password,
        port,
        file_list,
        basic_file_list,
        timestamp,
    ):
        try:
            if not os.path.exists(base_dir):
                print(f"Source directory '{base_dir}' does not exist.")
                return
            remote_root = os.path.join(remote_base, timestamp).replace("\\", "/")
            contents = os.listdir(base_dir)
            folders = [
                item for item in contents if os.path.isdir(os.path.join(base_dir, item))
            ]
            matching_files = basic_file_list
            for subfolder in folders:
                subfolder_path = os.path.join(base_dir, subfolder)
                sub_files_found = [
                        [subfolder_path, f]
                        for f in file_list
                        if os.path.isfile(os.path.join(subfolder_path, f))
                    ]
                if len(sub_files_found) == len(file_list):
                    matching_files.extend(sub_files_found)
            print(f"Creating tarball {timestamp}.tar.gz")
            tar_path = create_tarball(base_dir, matching_files, timestamp)
            for f in tar_path:
                if os.path.isfile(f):
                    print(f"Tarball created at: {f}")
                else:
                    print(f"Failed to create tarball at: {f}")
                print(f"Uploading tarball {f} to: {remote_root}")
                copy_file_with_smbclient(f, remote_root, host, username, password)
                print(f"Uploaded {f} to: {remote_root}")
            for f in tar_path:
                ssh_client = create_ssh_client(host, port, username, password)
                remote_ssh_folder = f"/Users/{remote_base}/{timestamp}".replace(
                    "\\", "/"
                ).replace("//", "/")
                untar_cmd = (
                    f'cd "{remote_ssh_folder}" && tar -xzf {os.path.basename(f)}'
                )
                stdin, stdout, stderr = ssh_client.exec_command(untar_cmd)
                exit_code = stdout.channel.recv_exit_status()
                if exit_code == 0:
                    print(f"Untarred successfully in remote: {remote_ssh_folder}")
                else:
                    print(
                        f"Untar failed in {remote_ssh_folder}:\n{stderr.read().decode()}"
                    )
                ssh_client.close()
        except Exception as e:
            print(f"An error occurred: {e}")

    def create_ssh_client(hostname, port, username, password):
        import paramiko

        client = paramiko.SSHClient()
        client.load_system_host_keys()
        client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        client.connect(hostname, port=port, username=username, password=password)
        return client

    def copy_file_from_remote(
        hostname, port, username, password, remote_path, local_path
    ):
        from scp import SCPClient

        try:
            ssh_client = create_ssh_client(hostname, port, username, password)
            with SCPClient(ssh_client.get_transport()) as scp:
                scp.get(remote_path, local_path)
            print(f"Copied '{remote_path}' to '{local_path}'")
        except Exception as e:
            print(f"An error occurred: {e}")

    def copy_file_from_remote_partial(
        hostname,
        port,
        username,
        password,
        remote_desktop_path,
        partial_name,
        local_path,
    ):
        from scp import SCPClient

        try:
            ssh_client = create_ssh_client(hostname, port, username, password)
            sftp = ssh_client.open_sftp()
            remote_files = sftp.listdir(remote_desktop_path)
            matching_files = [f for f in remote_files if partial_name in f]
            if not matching_files:
                print(
                    f"No files matching '{partial_name}' found in {remote_desktop_path}"
                )
                return
            os.makedirs(local_path, exist_ok=True)
            with SCPClient(ssh_client.get_transport()) as scp:
                for file_name in matching_files:
                    full_remote_path = os.path.join(remote_desktop_path, file_name)
                    local_file_path = os.path.join(local_path, file_name)
                    scp.get(full_remote_path, local_file_path)
                    print(f"Copied '{full_remote_path}' to '{local_file_path}'")
        except Exception as e:
            print(f"An error occurred: {e}")
        finally:
            if "sftp" in locals():
                sftp.close()
            if "ssh_client" in locals():
                ssh_client.close()

    def run_remote_script(
        host,
        port,
        username,
        password,
        remote_script_path,
        perf_testing,
        profile_perf,
        rel_err_pc,
        dtype,
    ):
        import paramiko

        client = paramiko.SSHClient()
        client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        try:
            client.connect(
                hostname=host, port=port, username=username, password=password
            )
            command = f"python {remote_script_path}"
            if perf_testing:
                command += " --perf_testing"
            if profile_perf:
                command += " --profile_perf"
            if rel_err_pc:
                command += " --rel_err_pc"
            if dtype:
                command += " --dtype-map dtype_map.json"
            stdin, stdout, stderr = client.exec_command(command)
            stdout.channel.recv_exit_status()
            print("Output:", stdout.read().decode())
            print("Errors:", stderr.read().decode())
        finally:
            client.close()

    def generate_output_path(output_dir, timestamp):
        return f"{output_dir}/output_{timestamp}.json"

    def remove_remote_directory(hostname, port, username, password, directory):
        try:
            ssh_client = create_ssh_client(hostname, port, username, password)
            command = f'rd /s /q "{directory}"'
            stdin, stdout, stderr = ssh_client.exec_command(command)
            print(stdout.read().decode())
            print(stderr.read().decode())
            ssh_client.close()
            print(f"Directory {directory} removed successfully.")
        except Exception as e:
            print(
                f"An error occurred while trying to remove the directory {directory} on {hostname}: {e}"
            )

    def copy_and_run_script(src_dir, script_name, des_dir):
        src_script_path = os.path.join(src_dir, script_name)
        shutil.copy(src_script_path, des_dir)
        des_script_path = os.path.join(des_dir, script_name)
        subprocess.run(["python", des_script_path], cwd=des_dir)

    def create_tarball(folder_path, file_list, tar_name):
        max_size_bytes = 1024 * 1024 * 1024
        archive_num = 0
        current_tar = None
        current_size = 0
        total_size = sum(os.path.getsize(os.path.join(f[0], f[1])) for f in file_list)
        print(f"Total size of files to be added: {total_size // 1024 // 1024} Mbytes")
        completed_size = 0
        tar_path = []
        completion_percentage = 0.0
        for f0, f1 in file_list:
            full_path = os.path.join(f0, f1)
            if os.path.isfile(full_path):
                item_size = os.path.getsize(full_path)
                if current_tar is None or current_size + item_size > max_size_bytes:
                    if current_tar:
                        current_tar.close()
                    archive_name = os.path.join(
                        tempfile.gettempdir(), f"{tar_name}_{archive_num}.tar.gz"
                    )
                    current_tar = tarfile.open(archive_name, "w:gz")
                    tar_path.append(archive_name)
                    current_size = 0
                    archive_num += 1
                tarinfo = current_tar.gettarinfo(
                    full_path,
                    arcname=os.path.join(os.path.basename(f0), os.path.basename(f1)),
                )
                completed_size += tarinfo.size
                current_size += item_size
                with open(full_path, "rb") as file_obj:
                    current_tar.addfile(tarinfo, file_obj)
            else:
                print(f"Skipping missing file: {full_path}")
            if (completed_size / total_size) > completion_percentage:
                completion_percentage += 0.1
                print(f"Progress: {completed_size / total_size:.0%} completed")
        if current_tar:
            current_tar.close()
        return tar_path

    def run_remote_script_golden_io(
        host, port, username, password, remote_script_path, subfolder, mode="copy"
    ):
        import paramiko

        client = paramiko.SSHClient()
        client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        try:
            client.connect(
                hostname=host, port=port, username=username, password=password
            )
            command = f"python {remote_script_path} {subfolder} {mode}"
            stdin, stdout, stderr = client.exec_command(command)
            stdout.channel.recv_exit_status()
            print("Output:", stdout.read().decode())
            print("Errors:", stderr.read().decode())
        finally:
            client.close()

    def _ensure_dir_local(p):
        os.makedirs(p, exist_ok=True)

    def _copy_selected_tree_local(src, dst_root, file_list_pairs, keep_rel=True):
        _ensure_dir_local(dst_root)
        for base, rel in file_list_pairs:
            src_path = os.path.join(base if base else "", rel)
            if not os.path.isfile(src_path):
                continue
            if keep_rel and base:
                rel_subdir = os.path.basename(os.path.dirname(src_path))
                dst_dir = os.path.join(dst_root, rel_subdir)
            else:
                dst_dir = dst_root
            os.makedirs(dst_dir, exist_ok=True)
            shutil.copy2(src_path, os.path.join(dst_dir, os.path.basename(src_path)))

    def _copy_output_subtrees_local(base_dir, dst_root, subfile_names):
        _ensure_dir_local(dst_root)
        for entry in os.scandir(base_dir):
            if not entry.is_dir():
                continue
            src_sub = entry.path
            present = [
                f for f in subfile_names if os.path.isfile(os.path.join(src_sub, f))
            ]
            if not present:
                continue
            dst_sub = os.path.join(dst_root, os.path.basename(src_sub))
            os.makedirs(dst_sub, exist_ok=True)
            for f in present:
                shutil.copy2(os.path.join(src_sub, f), os.path.join(dst_sub, f))

    def _run_local_script(py_path, perf_testing, profile_perf, rel_err_pc, dtype):
        cmd = ["python"] + py_path.split()
        if perf_testing:
            cmd.append("--perf_testing")
        if profile_perf:
            cmd.append("--profile_perf")
        if rel_err_pc:
            cmd.append("--rel_err_pc")
        if dtype:
            cmd += ["--dtype-map", "dtype_map.json"]
        subprocess.run(cmd, check=True)

    def _fetch_file_local(src_path, local_dest_path):
        os.makedirs(os.path.dirname(local_dest_path), exist_ok=True)
        shutil.copy2(src_path, local_dest_path)

    def _fetch_matching_local(folder, partial, local_dir):
        os.makedirs(local_dir, exist_ok=True)
        for name in os.listdir(folder):
            if partial in name:
                shutil.copy2(os.path.join(folder, name), os.path.join(local_dir, name))

    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    source_path = f"{HW_req}"
    destination_path = "/Administrator/Desktop/WAIC_test"
    remote_path = f"C:/Users/Administrator/Desktop/WAIC_test/{timestamp}/output.json"
    remote_path_2 = f"C:/Users/Administrator/Desktop/WAIC_test/{timestamp}/output.xlsx"
    basic_file_list = []
    print(f"HW IP: {host}")
    port = 22
    username = "Administrator"
    password = os.getenv("WAIC_CI_HW_PWD", "amdlabp@ssw0rd")
    LOCAL_WINDOWS = _is_windows_local()
    dest_root = os.path.join(
        "C:/Users/Administrator/Desktop/WAIC_test", timestamp
    ).replace("\\", "/")

    if rename:
        copy_and_run_script(source_path, "shape_rename_json.py", output)

    if not disable_fast_pm:
        copy_and_run_script(source_path, "collect_pm_ids_script.py", output)
        pm_id_list_path = os.path.join(output, "pm_id_list.json")
        if os.path.isfile(pm_id_list_path):
            basic_file_list.append(["", pm_id_list_path])

    copy_and_run_script(source_path, "collect_dtypes.py", output)
    dtype_list_path = os.path.join(output, "dtype_map.json")
    if os.path.isfile(dtype_list_path):
        basic_file_list.append(["", dtype_list_path])
        dtype = True

    if xclbin == 1:
        gen_xclbin(overlay, output, use_bsub)
        xclbin_path = output

    xclbin_list = ["out.xclbin"]
    if not disable_fast_pm:
        pm_files = glob.glob(os.path.join(xclbin_path, "pm_*.bin"))
        txn_files = glob.glob(os.path.join(xclbin_path, "txn_pm_*.bin"))
        xclbin_list += pm_files + txn_files
    for x in xclbin_list:
        xpath = os.path.join(xclbin_path, x)
        if os.path.exists(xpath):
            basic_file_list.append(["", xpath])
        else:
            raise ValueError(f"Error: Binary does not exist. {xpath}")

    basic_file_list += [
        ["", os.path.join(source_path, x)]
        for x in [
            "dolphin_test_ver4.py",
            "xrt_flow_test_patch_datatype_debug.exe",
            "xrt.ini",
            "copy_golden_io.py",
        ]
    ]

    if LOCAL_WINDOWS:
        os.makedirs(dest_root, exist_ok=True)
        _copy_selected_tree_local(
            src="",
            dst_root=dest_root,
            file_list_pairs=basic_file_list,
            keep_rel=False,
        )
        if golden_io is not None:
            if "update" in golden_io:
                golden_subfolders = [s for s in golden_io if s != "update"]
                if not golden_subfolders:
                    print("[ERROR] No subfolder specified for update mode.")
                    sys.exit(1)
                _copy_output_subtrees_local(
                    base_dir=output,
                    dst_root=dest_root,
                    subfile_names=[
                        "ifm.bin",
                        "ofm.bin",
                        "param.bin",
                        "txn.bin",
                        "wgt.bin",
                        "ctrl.bin",
                        "patch.json",
                    ],
                )
                local_copy_golden = os.path.join(dest_root, "copy_golden_io.py")
                for sub in golden_subfolders:
                    _run_local_script(
                        f"{local_copy_golden} {sub} update", False, False, False, False
                    )
            else:
                golden_subfolders = golden_io
                _copy_output_subtrees_local(
                    base_dir=output,
                    dst_root=dest_root,
                    subfile_names=[
                        "param.bin",
                        "txn.bin",
                        "wgt.bin",
                        "ctrl.bin",
                        "patch.json",
                    ],
                )
                local_copy_golden = os.path.join(dest_root, "copy_golden_io.py")
                for sub in golden_subfolders:
                    _run_local_script(
                        f"{local_copy_golden} {sub} copy", False, False, False, False
                    )
        else:
            _copy_output_subtrees_local(
                base_dir=output,
                dst_root=dest_root,
                subfile_names=[
                    "ifm.bin",
                    "ofm.bin",
                    "param.bin",
                    "txn.bin",
                    "wgt.bin",
                    "ctrl.bin",
                    "patch.json",
                ],
            )
        local_test_py = os.path.join(dest_root, "dolphin_test_ver4.py")
        print("executing dophin_test_ver4.py locally")
        _run_local_script(local_test_py, perf_testing, profile_perf, rel_err_pc, dtype)
        local_path = generate_output_path(output, timestamp)
        _fetch_file_local(os.path.join(dest_root, "output.json"), local_path)
        _fetch_file_local(
            os.path.join(dest_root, "output.xlsx"), local_path.replace(".json", ".xlsx")
        )
        if profile_perf:
            f = open(os.path.join(os.path.abspath(output), "rs_timer_result.log"), "w")
            remote_desktop_path = dest_root
            _fetch_matching_local(remote_desktop_path, "record_timer_ts", output)
            for file_path in [
                ft.path
                for ft in os.scandir(os.path.abspath(output))
                if "record_timer_ts" in ft.name and ft.name.endswith(".json")
            ]:
                try:
                    rt_ts = read_record_timer_ts(file_path)
                    print(
                        f"record_timer_ts {os.path.basename(file_path).ljust(70)}: {rt_ts}us"
                    )
                    f.write(
                        f"record_timer_ts {os.path.basename(file_path).ljust(70)}: {rt_ts}us\n"
                    )
                except Exception:
                    print(
                        f"Failed to collect record timer result for {os.path.basename(file_path)}"
                    )
                    f.write(
                        f"Failed to collect record timer result for {os.path.basename(file_path)}\n"
                    )
            f.close()
        shutil.rmtree(dest_root, ignore_errors=True)
        print(f"Directory {dest_root} removed successfully.")
    else:
        create_dir_with_smbclient(
            destination_path + "/" + timestamp, host, username, password
        )
        if golden_io is not None:
            if "update" in golden_io:
                golden_subfolders = [s for s in golden_io if s != "update"]
                if not golden_subfolders:
                    print("[ERROR] No subfolder specified for update mode.")
                    sys.exit(1)
                copy_and_untar_allfolders(
                    output,
                    destination_path,
                    host,
                    username,
                    password,
                    port,
                    [
                        "ifm.bin",
                        "ofm.bin",
                        "param.bin",
                        "txn.bin",
                        "wgt.bin",
                        "ctrl.bin",
                        "patch.json",
                    ],
                    basic_file_list,
                    timestamp,
                )
                remote_script_path = f"C:/Users/Administrator/Desktop/WAIC_test/{timestamp}/copy_golden_io.py"
                for subfolder in golden_subfolders:
                    run_remote_script_golden_io(
                        host,
                        port,
                        username,
                        password,
                        remote_script_path,
                        subfolder=subfolder,
                        mode="update",
                    )
            else:
                golden_subfolders = golden_io
                copy_and_untar_allfolders(
                    output,
                    destination_path,
                    host,
                    username,
                    password,
                    port,
                    ["param.bin", "txn.bin", "wgt.bin", "ctrl.bin", "patch.json"],
                    basic_file_list,
                    timestamp,
                )
                remote_script_path = f"C:/Users/Administrator/Desktop/WAIC_test/{timestamp}/copy_golden_io.py"
                for subfolder in golden_subfolders:
                    run_remote_script_golden_io(
                        host,
                        port,
                        username,
                        password,
                        remote_script_path,
                        subfolder=subfolder,
                        mode="copy",
                    )
        else:
            copy_and_untar_allfolders(
                output,
                destination_path,
                host,
                username,
                password,
                port,
                [
                    "ifm.bin",
                    "ofm.bin",
                    "param.bin",
                    "txn.bin",
                    "wgt.bin",
                    "ctrl.bin",
                    "patch.json",
                ],
                basic_file_list,
                timestamp,
            )
        remote_script_path = (
            f"C:/Users/Administrator/Desktop/WAIC_test/{timestamp}/dolphin_test_ver4.py"
        )
        print("executing dophin_test_ver4.py on remote machine")
        run_remote_script(
            host,
            port,
            username,
            password,
            remote_script_path,
            perf_testing,
            profile_perf,
            rel_err_pc,
            dtype,
        )
        local_path = generate_output_path(output, timestamp)
        print(f"dophin_test_ver4.py completed, copying output.json to {local_path}")
        copy_file_from_remote(host, port, username, password, remote_path, local_path)
        copy_file_from_remote(
            host,
            port,
            username,
            password,
            remote_path_2,
            local_path.replace(".json", ".xlsx"),
        )
        if profile_perf:
            with open(
                os.path.join(os.path.abspath(output), "rs_timer_result.log"), "w"
            ) as f:
                remote_desktop_path = (
                    f"C:/Users/Administrator/Desktop/WAIC_test/{timestamp}"
                )
                copy_file_from_remote_partial(
                    host,
                    port,
                    username,
                    password,
                    remote_desktop_path,
                    "record_timer_ts",
                    output,
                )
                for file_path in [
                    ft.path
                    for ft in os.scandir(os.path.abspath(output))
                    if "record_timer_ts" in ft.name and ft.name.endswith(".json")
                ]:
                    try:
                        rt_ts = read_record_timer_ts(file_path)
                        print(
                            f"record_timer_ts {os.path.basename(file_path).ljust(70)}: {rt_ts}us"
                        )
                        f.write(
                            f"record_timer_ts {os.path.basename(file_path).ljust(70)}: {rt_ts}us\n"
                        )
                    except Exception:
                        print(
                            f"Failed to collect record timer result for {os.path.basename(file_path)}"
                        )
                        f.write(
                            f"Failed to collect record timer result for {os.path.basename(file_path)}\n"
                        )
        remote_path_remove = f"C:/Users/Administrator/Desktop/WAIC_test/{timestamp}"
        remove_remote_directory(host, port, username, password, remote_path_remove)
