# fmt: on
import argparse
import subprocess
import os
import sys
from typing import List


def get_staged_files(matching_pattern: str) -> List[str]:
    staged_files = (
        subprocess.check_output(
            [
                "git",
                "diff",
                "--staged",
                "--name-only",
                "--diff-filter=d",
                "--",
                matching_pattern,
            ]
        )
        .decode()
        .split()
    )

    return staged_files


def get_changed_files_from_target(
    matching_pattern: str, target_branch: str
) -> List[str]:
    changed_files = (
        subprocess.check_output(
            [
                "git",
                "diff",
                target_branch,
                "--name-only",
                "--diff-filter=d",
                "--",
                matching_pattern,
            ]
        )
        .decode()
        .split()
    )

    return changed_files


def run_linter(python_files: list[str]) -> bool:
    """
    Run Python linter for passed Python files.
    Return True on success, False on error.
    """
    if not python_files:
        print(f"No python files found for running the linter")
        return True
    print(f"Running linter on files: {' '.join(python_files)}")
    try:
        subprocess.run(["pylint"] + python_files, check=True)
    except subprocess.CalledProcessError:
        return False
    return True


def filter_formatting_opt_in(python_files: list[str]) -> list[str]:
    """
    Filter list of passed Python file names to those that opt-in to
    auto-formatting by includeing "auto-format opt-in" anywhere in the file.
    """
    format_files: list[str] = []
    for python_file in python_files:
        with open(python_file, "r", encoding="utf-8") as py_f:
            if "# fmt: on" in py_f.read():  # note: this will opt-in this file
                format_files.append(python_file)
    return format_files


def run_format(python_files: list[str], check_only: bool, git_add: bool) -> bool:
    """
    Run (check of) automatic formatting of Python files that opt-in to
    auto-formatting.
    This function can also "git add" the formatted files.
    Return True on success, False on error.
    """
    format_files = filter_formatting_opt_in(python_files)
    if not format_files:
        return True
    check = ["--check", "--diff"] if check_only else []
    try:
        subprocess.run(["black"] + check + format_files, check=True)
    except subprocess.CalledProcessError:
        return False
    if git_add:
        subprocess.run(["git", "add"] + format_files, check=True)
    return True


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="Tool to run automation tools (formatter, linter) "
        "on a set of python files (staged file, diff against a target branch)"
    )

    parser.add_argument(
        "--staged",
        action="store_true",
        help="Run automation tools on all the python files which are staged",
    )
    parser.add_argument(
        "--target",
        help="Run automation tools on all the python files which differ from the target branch",
    )

    parser.add_argument(
        "--linter", action="store_true", help='Run Python linter "pylint"'
    )
    parser.add_argument(
        "--format",
        action="store_true",
        help='Automatically format using "black"',
    )
    parser.add_argument(
        "--check-format",
        action="store_true",
        help='Check formatting using "black --check"',
    )

    args = parser.parse_args()

    if args.staged and args.target:
        raise RuntimeError("staged and target option cannnot be used at the same time")
    if not args.staged and not args.target:
        raise RuntimeError("either staged or target option must be used")

    return args


def main() -> int:
    args = parse_args()

    # get the root of the repostory using the location of this file inside it
    repo_root_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))

    # enter the root of the repo:
    # - git diff find all files even with pattern matching
    # - pylint will use the configuration file located in there
    os.chdir(repo_root_path)

    # get the path of all staged python files (relative to the repo root)
    if args.staged:
        python_files = get_staged_files("*.py")
    elif args.target:
        python_files = get_changed_files_from_target("*.py", args.target)
    else:
        assert False, (
            "Unreachable, because parse_args enforces either staged or target."
            "Pylint needs this to detect that python_files is always defined."
        )

    success = True
    if args.linter:
        success &= run_linter(python_files)
    if args.format:
        success &= run_format(python_files, check_only=False, git_add=True)
    if args.check_format:
        success &= run_format(python_files, check_only=True, git_add=False)

    return 0 if success else 1


if __name__ == "__main__":
    sys.exit(main())
