import sys
import os
import argparse
import pandas as pd
import pdb
import subprocess
import filecmp
import logging


def run_cmd(cmd):
	try:
		result = subprocess.run(cmd, shell=True, check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
		return result.stdout.strip()
	except subprocess.CalledProcessError as e:
		print("Error:\n", e.stderr)
		return None

def compare_output(gold_file, is_update_golden):
	if os.path.exists(gold_file):
		result = filecmp.cmp("data_flow.py", gold_file, shallow=False)
	else:
		open(gold_file,'w').close()
		logging.info(f"Created file since it doesn't exist: {gold_file}")
		result = False

	if result is False:
		print(f"golden mismatch. {gold_file}")
		if is_update_golden:
			update_golden(gold_file)
		else:
			assert False, "Golden mismatch. Please check"

def update_golden(golden_file):
	logging.info(f"updating gold file:{golden_file}")
	command = f"cp data_flow.py {golden_file}"
	run_cmd(command)

def run_test_and_compare(input_json, golden_file, is_update_golden):
	command = f"python src/Scheduling_Engine/main.py -test {input_json}"
	run_cmd(command)
	compare_output(golden_file, is_update_golden)

def main(args):
	logging.info("Argument summary: %s", args)
	data = pd.read_csv(args.input_file)
	for index, row in data.iterrows():
		if not row['input_path'].startswith('#'):
			input_json = row['input_path']
			expected_output = row['golden_path']
			logging.info(f"Test {index}:")
			print(f"  Input: {input_json}, Expected Output: {expected_output}")
			run_test_and_compare(input_json, expected_output, args.update_golden)



if __name__ == "__main__":
	parser = argparse.ArgumentParser(description="Regression script to compare put with golden for test list",
																	 usage='use "%(prog)s --help" for more info', 
																	 formatter_class=argparse.RawTextHelpFormatter)

	#required knobs
	parser.add_argument('-test','--input_file',required=True, help="Run all test and compare with golden output")
	#optional knobs
	parser.add_argument('-d','--debug', help="Dump dbg log to 'dbg_log.txt'", action="store_true", default=False)
	parser.add_argument('-u','--update_golden', help="Update golden files for failing test", action="store_true", default=False)
	
	args = parser.parse_args()
	if args.debug:
		filename = args.debug_file_name
		verbose  = utils.DEBUG_VERBOSE.str2enum(args.verbose).value  #pylint: disable=undefined-variable 
		print("Saving debug log as :",os.getcwd()+"/"+args.debug_file_name)

		logging.basicConfig(
		filename=filename,
		filemode='w',
		format='[%(asctime)s,%(msecs)d] [%(levelname)s]: %(message)s',
		datefmt='%M:%H:%S',
		level=verbose
		)

	main(args)

