import logging
import os
import sys
import dataclasses


@dataclasses.dataclass()
class Context:
    output_dir: str
    debug: bool = False


@dataclasses.dataclass()
class Logger:
    # Name of the logger instance create by the logging module
    name: str

    # Context used to configure the logger
    context: Context

    # Logger instance
    _logger: logging.Logger = dataclasses.field(init=False)

    # Set to true when an NullLogger is needed
    _disabled: bool = False

    # Identation level from which the logging message will be printed
    _indentation_level: int = 0

    def init_null_logger(self):
        self._logger = logging.getLogger("NullLogger")
        if self._logger.handlers != 0:
            return

        self._logger.addHandler(logging.NullHandler())

    def __post_init__(self):
        """
        Initialize the logger instance with the relevant handler(s)
        """
        if self._disabled:
            self.init_null_logger()
            return

        self._logger = logging.getLogger(self.name)
        assert len(self._logger.handlers) == 0, "The logger should be a unconfigured instance"

        # Disable of the use of the parent logger's handlers and set logging level to debug
        # in order to accept all severity levels
        self._logger.propagate = False
        self._logger.setLevel(logging.DEBUG)

        # create console handler and set level to WARNING
        stream_handler = logging.StreamHandler(sys.stdout)
        stream_handler.setLevel(logging.INFO)
        self._logger.addHandler(stream_handler)

        # create a file handle and set level to DEBUG if debugging is enabled
        if self.context.debug:
            file_handler = logging.FileHandler(self.get_log_file(), mode="w")
            file_handler.setLevel(logging.DEBUG)
            self._logger.addHandler(file_handler)

    @staticmethod
    def get_null_logger() -> "Logger":
        context = Context("")
        return Logger("NullLogger", context, _disabled=True)

    def reset_indentation(self):
        self._indentation_level = 0

    def increase_indentation(self):
        if self._indentation_level < 0:
            self._indentation_level = 0
        self._indentation_level += 1

    def decrease_indentation(self):
        if self._indentation_level <= 0:
            self._indentation_level = 0
        else:
            self._indentation_level -= 1

    def _prepend_indentation(self, msg: str) -> str:
        return (" " * self._indentation_level * 4) + msg

    def debug(self, msg, *args, **kwargs):
        msg = self._prepend_indentation(msg)
        self._logger.debug(msg, *args, **kwargs)

    def warning(self, msg, *args, **kwargs):
        msg = self._prepend_indentation(msg)
        self._logger.warning(msg, *args, **kwargs)

    def info(self, msg, *args, **kwargs):
        msg = self._prepend_indentation(msg)
        self._logger.info(msg, *args, **kwargs)

    def get_log_file(self) -> str:
        return os.path.join(self.context.output_dir, self.name + ".log")

    def __del__(self):
        # Close all handlers when destructing the instance in order to have
        # a clean instance if reused before shutdown of the logging module.
        for handler in self._logger.handlers[:]:
            handler.close()
            self._logger.removeHandler(handler)
