---
applyTo: "**"
---

# aie4_models Copilot Instructions

Before running the python scripts in this repository, please ensure you have completed the following setup steps:

1. **Source the Environment Variables:**

   - Open your bash terminal.
   - Navigate to the root directory of the `aie4_models` repository.
   - Run the following command to source the environment variables:
     ```bash
     source settings.sh && source env/bin/activate
     ```

2. **Always use unix-style forward slashes in paths:**

   - When specifying file paths in your scripts or commands, ensure you use unix-style forward slashes (`/`). This is important for compatibility with the scripts in this repository.
   - Use forward slashes in all file paths
     - Example: `utils/data/input_data.txt`
     - Non-Example: `utils\data\input_data.txt`

3. **Editing files:**

   - When suggesting edits to files, make sure to follow the existing code style and conventions used in the repository. This includes indentation, naming conventions, and commenting style.
   - Always give an option to accept or undo changes when suggesting edits.
   - Always show code changes inline in the code-editor or chat window.
   - **Check for existing utilities first** - Before writing new code, search for similar functionality in the codebase using `grep_search` or `semantic_search`
   - **Single source of documentation** - Don't duplicate detailed implementation docs across files; reference the authoritative source instead

4. **Follow coding conventions:**

   - Adhere to the coding standards and best practices established in the repository. This includes writing clean, maintainable code and providing appropriate documentation for any new functions or classes you introduce.
   - Ensure that there are no pylint or flake8 errors or warnings in the code you suggest.
   - Read `lint.sh` to understand the flake8 and pylint args that we use
   - Keep python imports at the begining of file. No imports inside the function. Follow conventions e.g., builtin libraries imports come first before third-party imports.
   - We are writing a production code.

5. **You are a senior staff engineer - Don't write crappy code**

   - **NEVER duplicate logic** - If utility functions exist, USE them. Don't reimplement manually.
   - **SEARCH before implementing** - Use `grep_search` to find existing utilities (e.g., `tensor_dtype_to_size`, `get_padded_shape_rev`)
   - **DRY principle** - Don't Repeat Yourself. One source of truth for implementation details.
   - **Use type-safe utilities** - Don't manually calculate dtype sizes or padding; use existing functions that handle all cases
   - **Avoid magic numbers** - Use named constants and utility functions instead of hardcoded calculations like `((c + 63) // 64) * 64`
   - **Production quality** - Code should be maintainable, testable, and reusable
   - **No shortcuts** - Take the time to do it right the first time
   - **NO redundant comments** - Don't document implementation details that are already clear from the code or encapsulated in called functions
   - **Comments should explain WHY, not WHAT** - If the function/method name clearly states what it does, don't repeat it in a comment
   - **Don't duplicate documentation** - If validation logic is in a Pydantic model or factory method, don't re-document the validation rules in calling code
   - **Self-documenting code** - Use clear function names, type hints, and docstrings; avoid inline comments that just restate the code
   - **NEVER write monolithic functions** - Break down large functions into smaller, focused, reusable helpers
   - **Extract reusable logic** - If you see repeated patterns or chunks of logic that could be used elsewhere, extract them into utility functions
   - **Avoid manual string manipulation** - Don't manually build complex strings or regex patterns when libraries like `pathlib.PurePosixPath.match()` exist
   - **Think before implementing** - Question manual implementations: "Is there a stdlib or existing utility for this?" Examples:
     - Pattern matching: Use `fnmatch`, `pathlib.match()`, or `glob` instead of manual regex conversion
     - File operations: Use `Path.glob()` instead of `os.listdir()` + filtering
     - String formatting: Use f-strings or `.format()` instead of manual concatenation
     - Logging: Use lazy evaluation not f-strings for performance (e.g., `logger.debug("Message %s", value)` instead of `logger.debug(f"Message {value}")`)
   - **Avoid hardcoding values** - Use constants or configuration files instead of hardcoded values
   - **Use existing utilities** - Don't reinvent the wheel; use existing functions for common tasks
   - **Use parameterized tests** - Instead of writing multiple similar test cases, use `pytest.mark.parametrize` to run the same test with different inputs
   - **Avoid hardcoding values** - Use constants or configuration files instead of hardcoded values
   - **Avoid unnecessary complexity** - Keep code simple and clean; it's harder to write simple code. it's harder to maintain complex code
   - **Scalability matters** - If you hardcode N cases inline, adding case N+1 requires modifying the function. Make it data-driven instead.
   - **Generic over specific** - Write parameterized, reusable code rather than copy-pasting with minor variations

6. **Writing tests (follow industry best practices)**:

   - **DO NOT test enum values or language features** - Enums are Python constructs; test usage in business logic, not the enum definition itself
   - **DO NOT test library internals** - Don't test Pydantic/FastAPI/library behavior (e.g., extra field handling); test your validation rules
   - **DO NOT use artificial operations with empty inputs/outputs** - Never use `Operation(inputs=[], ...)` or `Operation(..., outputs=[])` in tests; real ONNX operations always have both inputs and outputs
   - **USE parametrization** - Consolidate similar tests with `@pytest.mark.parametrize` to reduce duplication
   - **TEST exception types, not exact messages** - Verify `ValidationError` is raised and check for key terms (e.g., "axis=0"), not exact message format
   - **FOCUS on behavior** - Test what your code does, not how the underlying libraries work
   - **USE fixtures** - Create reusable test data with pytest fixtures for DRY principles
   - **VERIFY error handling** - Test both happy paths and error conditions
   - **AVOID testing implementation details** - Test the public interface, not internal helper methods unless critical
   - **USE realistic test data** - Test operations should reflect actual ONNX graph patterns; mark graph I/O tensors with `is_model_io=True` instead of using artificial boundary operations

7. **Understanding tensor shapes and runtime operations**:

   - **All tensor shapes are implicitly NHWC** - The graph preprocessing converts ONNX models to NHWC format, but it's "sloppy" - it may output 3D or 4D shapes
   - **Implicit 4D interpretation** - A shape `(4, 10, 10)` means `(N=1, H=4, W=10, C=10)` in NHWC; a shape `(2, 4, 10, 10)` is explicit `(N=2, H=4, W=10, C=10)`
   - **Always normalize before calculations** - Use `normalize_shape_to_4d()` to handle both 3D and 4D inputs consistently
   - **Size per batch calculation** - After normalizing to 4D, `normalized_shape[1:]` gives `(H, W, C)`, which is the size of one batch element
   - **Padding is applied to C dimension** - Use `tensor.size` which accounts for padded channels when `is_channel_multiple_of_64=True`

8. **Git commit message conventions**:

   - **NO markdown formatting** - Do not use markdown emphasis shortcuts (\*_, _, \_, etc.) in commit messages
   - **Use plain text** - Commit messages should be readable in plain text terminals and git logs
   - **Structure** - Use conventional commit format: `type(scope): summary` followed by blank line and detailed body
   - **Sections in body** - Use plain text headers like "Changes:", "Benefits:", "Technical details:" (no bold/asterisks)
   - **Lists** - Use simple hyphens (-) or asterisks (\*) for bullet points without any bold/italic formatting
   - **Examples of GOOD commit messages**:

     ```
     feat(graph): Add concat/split runtime memory aliasing

     Implement memory-efficient concat_runtime and split_runtime operations.

     Changes:
     - Added allocate_split_runtime() method
     - Added allocate_concat_runtime() method
     - Integrated into scheduler with three-pass architecture
     ```

   - **Examples of BAD commit messages** (contains markdown formatting):

     ```
     feat(graph): Add concat/split runtime memory aliasing

     **Changes:**
     - Added allocate_split_runtime() method
     - **Core Features:** memory-efficient layout
     ```

9. **Git commit workflow protocol**:

   - **ALWAYS follow this protocol when committing changes**:
     1. Run `git status` to see unstaged changes
     2. Run `git diff` to review all changes in detail
     3. Stage related changes together: `git add <files>`
     4. Create logical, atomic commits - group related changes
     5. Follow commit message conventions (see section 8)
     6. Use multiple commits if changes serve different purposes
   - **Before committing**:
     - Review the full diff to understand all changes
     - Group changes by logical purpose (e.g., feature, fix, refactor)
     - Ensure each commit is self-contained and builds successfully
   - **Commit message quality**:
     - Write clear, descriptive summaries
     - Include context in the body explaining WHY the change was made
     - Reference related issues or PRs when applicable
   - **After pushing**
     - Create a PR request markdown message following the PR template

10. **GitHub Actions workflow development**:

- **GitHub Enterprise (GHE) environment** - This repo uses GitHub Enterprise, not github.com. Newer action versions may not be supported.
- **Consult existing workflows first** - Always check existing workflows in `.github/workflows/` to understand what action versions and patterns work in this GHE instance
- **Action version compatibility** - Use action versions that match existing workflows (e.g., `actions/upload-artifact@v4`). Don't blindly upgrade to latest versions.
- **Understand workflow architecture** - Check if workflows are coordinated by a parent workflow (like ci.yml) before adding triggers
- **Avoid duplicate triggers** - If a parent workflow already has pull_request triggers and calls child workflows, DO NOT add pull_request triggers to child workflows
- **Use workflow_call for reusable workflows** - Child workflows should only have workflow_call and workflow_dispatch triggers
- **Path filtering belongs in orchestrator** - Path-based conditional execution should be in the parent/orchestrator workflow, not individual child workflows
- **Test Python version compatibility** - GitHub Actions runners may have different Python versions; always specify python3 explicitly or use setup-python-env
- **Avoid shell-specific Python invocation** - Don't use `shell: python` (defaults to Python 2); use `shell: bash` with `python3 << 'EOF'` heredoc
- **Use existing actions** - Check for existing custom actions (like setup-python-env) before writing inline scripts
- **Multiline outputs** - Use heredoc syntax for multiline GitHub Actions outputs: `variable<<EOF ... EOF`
- **Generic over specific** - Create reusable, parameterized actions rather than hardcoded, workflow-specific ones
- **No third-party actions without approval** - GitHub Enterprise may restrict third-party actions; create custom alternatives instead
- **Don't use actions/upload-artifact@v4** - This version has issues in GHE; use v3 as in existing workflows

11. **Markdown formatting**:

- **MD040/fenced-code-language** - Fenced code blocks must always have a language specified
- Use appropriate language identifiers: `python`, `bash`, `yaml`, `json`, `typescript`, etc.
- Use `text` if the contents are not language-specific (e.g., plain output, logs, generic examples)
- Example of GOOD:
  ```text
  This is plain text output
  ```
- Example of BAD:
  ```
  This code block has no language
  ```

12. **GitHub PR description conventions**:

- Always write github PR messages following PR template (`.github/pull_request_template.md`).
