# Contributing to AIE4 Models

This document provides guidelines and instructions for contributing to this repository.

## Table of Contents

- [Getting Started](#getting-started)
- [Development Environment Setup](#development-environment-setup)
- [Code Standards](#code-standards)
- [Development Workflow](#development-workflow)
- [Testing Guidelines](#testing-guidelines)
- [Submitting Changes](#submitting-changes)
- [Adding New Operators](#adding-new-operators)

## Getting Started

### Prerequisites

- Linux environment (RHEL/CentOS recommended)
- Python 3.8+
- GCC 12.3.0 or later
- Access to Xilinx Vitis tools
- Git with submodule support

### Repository Structure

```
aie4_models/
├── buildscripts/     # Build scripts for operators
├── buildtest/        # Regression test suite
├── cert_sim/         # CERT simulation infrastructure
├── dmacompiler/      # DMA compiler submodule
├── graph/            # Graph partitioning and memory allocation
├── host/             # Host-side simulation test code
├── kernel/           # AIE kernel implementations
├── kerneltest/       # Single-core kernel tests
├── scheduler/        # Operator scheduling and tiling
├── tiler/            # Tiling algorithms
├── utils/            # Common utilities
└── build_aie4.py     # Main build orchestration script
```

## Development Environment Setup

1. **Clone the repository with submodules:**

   ```bash
   git clone --recursive <repository-url>
   cd aie4_models
   ```

2. **Set up the environment:**

   ```bash
   source settings.sh
   ```

3. **Create and activate Python virtual environment:**

   ```bash
   python -m venv env
   source env/bin/activate
   pip install -r requirements.txt
   ```

4. **Install git hooks:**

   ```bash
   bash git-hooks/install-hooks.sh
   ```

## Code Standards

### Python Code

All Python code must pass the following linters:

- **flake8** - PEP 8 style guide enforcement
- **pylint** - Code analysis and quality checks
- **mypy** - Static type checking

Run linters before committing:

```bash
sh lint.sh py
```

#### Python Style Guidelines

- Follow PEP 8 conventions
- Use type hints for all function signatures
- Maximum line length: 120 characters
- Use docstrings for all public functions and classes
- Prefer f-strings over `.format()` or `%` formatting
- Use logger from `graph.utilities` for logging.

Example:

```python
def process_shape(
    data: Union[dict, list],
    backend: BackEnd = BackEnd.Adf,
    kernel_names: Optional[Sequence[str]] = None
) -> list[str]:
    """
    Process operator shape configuration.

    Args:
        data: Shape dimensions as dict or list
        backend: Target backend for compilation
        kernel_names: Optional kernel function names

    Returns:
        List of generated file paths
    """
    # Implementation
    pass
```

### Commit Messages

Follow conventional commit format:

```
<type>(<scope>): <subject>

<body>

<footer>
```

Types:

- `feat`: New feature
- `fix`: Bug fix
- `docs`: Documentation changes
- `style`: Code style changes (formatting, etc.)
- `refactor`: Code refactoring
- `test`: Adding or updating tests
- `chore`: Maintenance tasks

Example:

```
feat(gemm): add int4 weight support for L3 dataflow

Implement int4 quantization support for GEMM operator in L3
dataflow mode. Includes kernel updates and corresponding
build script changes.

Closes #123
```

## Development Workflow

### 1. Create a Feature Branch

```bash
git checkout -b feature/your-feature-name
# or
git checkout -b fix/issue-description
# or
git checkout -B fix/issue-description origin/main
# or
git checkout -B fix/issue-description <your_fork>/main
```

### 2. Make Changes

- Write code following the style guidelines
- Add/update tests as needed
- Update documentation

### 3. Run Quality Checks

```bash
# Run linters
sh lint.sh py

# Run regression tests
bash dma_check.sh

# Run unit tests
bash pytests.sh
```

### 4. Commit Changes

```bash
git add .
git commit -m "feat(scope): description"
```

### 5. Push and Create Pull Request

```bash
git push <your_fork> feature/your-feature-name
```

Then create a pull request on the repository platform.

## Testing Guidelines

### Continuous Integration

All pull requests must:

- Pass all linting checks (`lint.sh`)
- Pass unit tests (`pytests.sh`)
- Pass dma.hpp tests (`dma_check.sh`)
- Have no merge conflicts with main branch

### Pytests

#### To debug a specific test

Use

```bash
pytest -m <pytest_marker> -k <test_name> -s -vv -n 0 -o log_cli=true
```

e.g.,

```bash
pytest -m graph -k test_L2L3_allocator_psu0_subgraph_continuous -s -vv -n 0 -o log_cli=true
```

### Large-Scale Regression Testing

We have an automation script that allows op-developers to submit hundreds of parameterized tests (e.g., SIM tests) to LSF in seconds - each runs in parallel on the cluster instead of sequentially on your local machine. This saves a large amount of time for regression testing. See README_pytests.md for more information.

## Submitting Changes

### Pull Request Checklist

Before submitting a pull request, ensure:

- [ ] Code follows style guidelines
- [ ] All linters pass (`bash lint.sh py`)
- [ ] Regression tests pass (`bash dma_check.sh`)
- [ ] Unit tests pass (`bash pytests.sh`)
- [ ] New features have corresponding tests
- [ ] Documentation is updated
- [ ] Commit messages follow conventions
- [ ] Branch is up to date with main
- [ ] You have reviewed [Adding New Operators](adding-new-operators) in CONTRIBUTING.md

### Review Process

1. Submit pull request with clear description
2. Address automated CI feedback
3. Respond to reviewer comments
4. Make requested changes
5. Obtain approvals
6. Merge when approved

## Getting Help

- Check existing documentation in `README.md` and component READMEs
- Review similar implementations in the codebase
- Ask questions in pull request discussions
- Contact maintainers for guidance

## Adding New Operators

For every new operator (let's call it `abc`) added to the repository, the developer is responsible for making the following changes and following the naming conventions:

1. **Add the dataflow file** in the path: `dataflow/abc.py`.
2. **Add the tiler file** in the path: `tiler/abc_tiler.py`.
3. **Add testbench files**:
   - `host/abc.cpp`
   - `host/abc.hpp`
4. **Add the build script** for the operator in the path: `buildscripts/build_abc.py`. Follow the guidelines mentioned in the README of the `buildscripts` directory.
5. **Add the regression testing script** for the operator in the path: `buildtest/test_abc.py`. Follow the guidelines mentioned in the README of the `buildtest` directory.

Following these steps ensures that the new operator is correctly integrated into the repository and can be compiled via `build.py`.

### Follow these guidelines for developing operators

- Operators should be tightly integrated to read model path and attributes from corresponding Json files
- Operators Should handle sign bits, QDQ flags, etc and pass them through layer params or weights.
- Operators should generate the required ASM, and ELF files through the build system.
- Make sure to add pytests and integrate them to CI and Cron job scripts
- Run regression tests when appropriate, see README_pytests.md for more.
