# Coverage Testing

This directory contains coverage testing infrastructure for the `graph` module.

## Quick Start

Run coverage tests locally:

```bash
# Run coverage for all graph modules
bash coverage.sh
```

## Coverage Threshold

- **Minimum Required:** 95%

The CI pipeline will fail if coverage drops below 95%.

## Configuration Files

### `.coveragerc`

Main coverage configuration:

- `fail_under = 95` - Minimum required coverage percentage
- `show_missing = True` - Show line numbers of uncovered code
- `omit` - Files to exclude from coverage:
  - Test files: `*/tests/*`, `*/test_*.py`, `**/conftest.py`
  - Legacy files: `*/graph/resnet50.py`, `*/graph/clean_onnx_graph.py`
- `exclude_lines` - Patterns for code that doesn't need coverage (pragmas, abstract methods, etc.)

### `coverage.sh`

Wrapper script that:

- Disables beartype (via `DISABLE_BEARTYPE=1`) to avoid circular import issues during coverage collection
- Runs all graph tests with coverage: `pytest -m graph --cov=graph --cov-report=term-missing -v`
- Runs tests in parallel (6 workers via pytest-xdist)
- Excludes test files and legacy code via `.coveragerc` configuration

### `graph/__init__.py`

Conditional beartype loading:

```python
if os.getenv("DISABLE_BEARTYPE") != "1":
    from beartype.claw import beartype_this_package
    beartype_this_package()
```

## CI Integration

### Coverage Workflow (`.github/workflows/coverage.yml`)

Runs automatically on:

- Pull requests to `main`
- Pushes to `main`

The workflow:

1. **Runs `coverage.sh`** - Uses the same script developers run locally
2. Captures output and parses coverage percentage
3. Copies HTML and XML reports to `ci-coverage/` for artifact upload
4. Posts coverage results as PR comment
5. Uploads coverage artifacts (HTML report, XML, logs)
6. **Fails the build** if coverage < 95%

**Key benefit:** Developers can run the exact same command locally as CI runs:

```bash
# Same command runs in CI and locally
bash coverage.sh
```

No need to install `act` or other tools to reproduce CI coverage runs!

## Test Structure

Tests are organized under `graph/tests/` with one test file per module:

Each test file contains multiple test classes, typically one per function/class being tested, plus integration tests.

To see current test count and structure:

```bash
# List all graph tests
pytest graph/tests/ -m graph --collect-only

# Count tests per file
pytest graph/tests/test_multibin_allocator.py --collect-only
```

## Adding New Modules to Coverage

1. Create test file: `graph/tests/test_<module_name>.py`
2. Add `@pytest.mark.graph` to test classes
3. Write comprehensive tests targeting 95%+ coverage
4. Run coverage: `bash coverage.sh`

The coverage script automatically includes all graph modules - no configuration changes needed!

## Viewing Coverage Reports

### Terminal Output

Coverage summary is shown after test run with missing line numbers.

### HTML Report

```bash
# After running coverage.sh, open HTML report
xdg-open htmlcov/index.html  # Linux
open htmlcov/index.html      # macOS
```

The HTML report provides:

- Line-by-line coverage highlighting
- Per-file coverage percentages
- Branch coverage analysis
- Missing line numbers

### CI Artifacts

Download from GitHub Actions workflow run:

1. Go to Actions tab
2. Click on coverage workflow run
3. Download "coverage-report" artifact

## Troubleshooting

### Beartype Circular Import Issues

If you see circular import errors with beartype during coverage:

- Ensure `DISABLE_BEARTYPE=1` is set when running coverage
- Use `coverage.sh` script instead of running pytest directly
- The script automatically handles this for you

### Coverage Not Meeting Threshold

1. Check which lines are missing coverage:

   ```bash
   bash coverage.sh
   # Look for "Missing" column in output
   ```

2. Add tests for uncovered code
3. Re-run coverage to verify

Note: Overall coverage is currently 85% due to legacy code. Focus on keeping new/refactored modules at 95%+.

### Test Files Appearing in Coverage Report

If you see test files in the coverage report:

- Check that `.coveragerc` has proper `omit` patterns
- Ensure patterns include: `*/tests/*`, `*/test_*.py`, `**/conftest.py`
- Test files should NOT count toward coverage metrics

### Parallel Execution Issues

If tests fail intermittently:

- Check for shared state between tests
- Ensure tests are properly isolated
- Use fixtures with appropriate scope
- Consider using `@pytest.mark.serial` for tests that cannot run in parallel
