# Hardware E2E Testing

End-to-end testing infrastructure for running model tests on Windows machines with AMD NPU hardware.

## Support matrix

Following machines are currently supported:

```text
Medusa: 10.228.203.217, 10.228.200.62
Strix: 10.228.200.117
```

Strix machine is only used for compilation. The Medusa machine is generally used for execution.

## Quick Start

```bash
# Test a model on origin/main
python HW_requirements/e2e_test.py --model psd1

# Test a specific commit on origin/main
python HW_requirements/e2e_test.py --model psd1 --commit abc123

# Test a PR (compile on Strix, run on Medusa)
python HW_requirements/e2e_test.py --model psd1 --pr 123 --host 10.228.200.117 --skip-execute
python HW_requirements/copy_artifacts.py --model psd1 --via-ssh 10.228.200.117 10.228.203.217
python HW_requirements/e2e_test.py --model psd1 --pr 123 --host 10.228.203.217 --skip-compile

# Retest a PR (No compilation, just re-run)
python HW_requirements/e2e_test.py --model psd1 --pr 123 --skip-compile
```

## Supported Models

- `psd1`
- `psd2`
- `psd3`
- `psd4`
- `psh`
- `psi`
- `psu0`
- `psu1`
- `vit-base`

## CLI Options

| Option           | Description                                                            |
|------------------|------------------------------------------------------------------------|
| `--model`        | Model to test (psd1, psd2, psd3, psd4, psh, psi, psu0, psu1, vit-base) |
| `--host`         | Windows machine IP (default: 10.228.203.217)                           |
| `--commit`       | Git commit SHA to checkout                                             |
| `--pr NUMBER`    | PR number to test (fetches refs/pull/NUMBER/head)                      |
| `--skip-sync`    | Skip git sync, use current state on Windows                            |
| `--skip-compile` | Skip compilation, run pre-compiled only                                |
| `--skip-execute` | Skip execution, compile only (for split workflow)                      |
| `--output`       | Output markdown file path                                              |

## Split Compile/Run Workflow

For medusa machines that crash during compilation, it is advisable to compile first on strix machine and then run on medusa:

```bash
# Step 1: Compile only (large memory machine)
python HW_requirements/e2e_test.py --host 10.228.200.117 --model psd1 --skip-execute

# Step 2: Copy artifacts (direct Windows-to-Windows transfer)
# Uses --via-ssh to run rclone directly on source machine (faster, no intermediate hop)
python HW_requirements/copy_artifacts.py --model psd1 --via-ssh 10.228.200.117 10.228.203.217

# Step 3: Run only (target hardware)
python HW_requirements/e2e_test.py --host 10.228.203.217 --model psd1 --skip-compile
```

## Developer Workflow

### Option 1: Fast Iteration with `--skip-sync`

For rapid development cycles, manually control the code on Windows and skip git sync:

```bash
# 1. SSH to Windows and checkout your branch
ssh administrator@10.228.203.217
cd C:\ProgramData\miniforge3\envs\ci_etoe\Lib\site-packages\aie4_models
git remote add <username> https://gitenterprise.xilinx.com/<username>/aie4_models.git
git fetch <username>
git checkout <username>/my-feature-branch

# 2. From Linux, run tests (skip git sync)
python HW_requirements/e2e_test.py --model psd1 --skip-sync

# 3. Edit code on Windows, re-run
python HW_requirements/e2e_test.py --model psd1 --skip-sync
```

**When to use:** Fast iteration during development. You control the code state on Windows.

### Option 2: PR Validation with `--pr`

After creating a PR, validate it on hardware before merge:

```bash
# 1. Push your branch to your fork
git push <username> my-feature-branch

# 2. Create PR on gitenterprise (fork -> IPSP/aie4_models)

# 3. Run tests against the PR
python HW_requirements/e2e_test.py --model psd1 --pr 123
```

This fetches `refs/pull/123/head` from origin and runs tests against it.

**When to use:** Final validation before merge. Ensures the exact PR code is tested.

### Workflow Comparison

| Workflow       | Use Case        | Git State      | Requires Push    |
|----------------|-----------------|----------------|------------------|
| `--skip-sync`  | Fast iteration  | Manual control | No               |
| `--pr NUMBER`  | PR validation   | PR head        | Yes (PR created) |
| `--commit SHA` | Specific commit | Exact commit   | Yes (pushed)     |
| (default)      | CI/nightly      | origin/main    | No               |

## Troubleshooting

### Connection Issues

```bash
# Verify server is reachable
ping -c1 10.228.203.217

# Verify SSH works
ssh administrator@10.228.203.217 "echo ok"
```

### Test Failures

```bash
# Re-run with pre-compiled (skip compilation)
python HW_requirements/e2e_test.py --model psd1 --skip-compile

# Check report manually
ssh administrator@10.228.203.217
type C:\Users\Administrator\Desktop\shajaisw\psd1\report.csv
```

### Git Sync Failures

```bash
# Skip sync and debug manually
python HW_requirements/e2e_test.py --model psd1 --skip-sync

# SSH and check git state
ssh administrator@10.228.203.217
cd C:\ProgramData\miniforge3\envs\ci_etoe\Lib\site-packages\aie4_models
git status
git log -1
```
