"""Collect instance IDs from config.json files in subdirectories."""
import json
import os
import sys


def _ensure_list(x):
    """Return x as a list ([] if None)."""
    if x is None:
        return []
    if isinstance(x, list):
        return x
    return [x]


def collect_instance_ids(input_dir, output_dir):
    """Collect instance IDs from config files and save to JSON."""
    result = []

    for subfolder in os.listdir(input_dir):
        subfolder_path = os.path.join(input_dir, subfolder)
        if not os.path.isdir(subfolder_path):
            continue

        cfg_path = os.path.join(subfolder_path, "config.json")
        instance_ids = []

        if os.path.isfile(cfg_path):
            try:
                with open(cfg_path, "r", encoding="utf-8") as f:
                    data = json.load(f)

                instances = data.get("instance") or data.get("instances")

                for inst in _ensure_list(instances):
                    if isinstance(inst, dict) and "id" in inst:
                        instance_ids.append(str(inst["id"]))
            except Exception as e:  # pylint: disable=W0718
                print(f"Error reading {cfg_path}: {e}")

        # Clean duplicates and preserve order
        seen = set()
        ids = [i for i in instance_ids if not (i in seen or seen.add(i))]

        # Always add entry, even if empty
        result.append({
            "folder": subfolder,
            "instance_id": ids if ids else ["aie4_models"]
        })

    os.makedirs(output_dir, exist_ok=True)
    output_file = os.path.join(output_dir, "instance_ids_list.json")
    with open(output_file, "w", encoding="utf-8") as f:
        json.dump(result, f, indent=2, ensure_ascii=False)

    print(f"Saved {len(result)} entries to {output_file}")


if __name__ == "__main__":
    script_dir = os.path.dirname(os.path.abspath(__file__))
    if len(sys.argv) == 3:
        in_dir = os.path.abspath(sys.argv[1])
        out_dir = os.path.abspath(sys.argv[2])
    else:
        print("No input/output directories provided — using script location as default.")
        in_dir = out_dir = script_dir

    collect_instance_ids(in_dir, out_dir)
