# Bootstrap script for Windows machine setup
# Run this ON the Windows machine as Administrator
# This script sets up the machine for E2E testing
#
# Usage:
#   .\bootstrap_machine.ps1 -PublicKey "ssh-rsa AAAA..."
#   .\bootstrap_machine.ps1  # Will prompt for public key

param(
    [Parameter(Mandatory=$false)]
    [string]$PublicKey
)

$ErrorActionPreference = "Stop"

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Windows Machine Bootstrap Script" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan

# Prompt for public key if not provided
if (-not $PublicKey) {
    Write-Host "`nNo public key provided. Get it from your Linux machine:" -ForegroundColor Yellow
    Write-Host "  cat ~/.ssh/id_rsa.pub" -ForegroundColor Gray
    Write-Host ""
    $PublicKey = Read-Host "Paste your SSH public key (ssh-rsa ...)"
    if (-not $PublicKey -or -not $PublicKey.StartsWith("ssh-")) {
        Write-Host "Invalid public key. Must start with 'ssh-rsa' or 'ssh-ed25519'" -ForegroundColor Red
        exit 1
    }
}

# Step 0: Set execution policy to allow scripts
Write-Host "`n[0/5] Setting execution policy..." -ForegroundColor Yellow
Set-ExecutionPolicy -ExecutionPolicy Bypass -Scope LocalMachine -Force
Set-ExecutionPolicy Bypass -Scope Process -Force
Set-ExecutionPolicy Bypass -Scope CurrentUser -Force
Write-Host "Execution policy set to Bypass" -ForegroundColor Green

# Step 1: Install Chocolatey if not present
Write-Host "`n[1/5] Checking Chocolatey installation..." -ForegroundColor Yellow
if (!(Get-Command choco -ErrorAction SilentlyContinue)) {
    Write-Host "Installing Chocolatey..." -ForegroundColor Green
    Set-ExecutionPolicy Bypass -Scope Process -Force
    [System.Net.ServicePointManager]::SecurityProtocol = [System.Net.ServicePointManager]::SecurityProtocol -bor 3072
    Invoke-Expression ((New-Object System.Net.WebClient).DownloadString('https://community.chocolatey.org/install.ps1'))

    # Refresh PATH
    $env:Path = [System.Environment]::GetEnvironmentVariable("Path", "Machine") + ";" + [System.Environment]::GetEnvironmentVariable("Path", "User")
    Write-Host "Chocolatey installed successfully" -ForegroundColor Green
} else {
    Write-Host "Chocolatey already installed: $(choco --version)" -ForegroundColor Green
}

# Step 2: Install tools via Chocolatey
Write-Host "`n[2/9] Installing vim, rclone, git..." -ForegroundColor Yellow
choco install vim rclone git -y --no-progress
Write-Host "Vim, rclone, and git installed" -ForegroundColor Green

# Step 3: Configure Git credential manager
Write-Host "`n[3/9] Configuring Git credential manager..." -ForegroundColor Yellow
# Refresh PATH to pick up git
$env:Path = [System.Environment]::GetEnvironmentVariable("Path", "Machine") + ";" + [System.Environment]::GetEnvironmentVariable("Path", "User")
git config --global credential.helper manager
Write-Host "Git credential manager configured" -ForegroundColor Green

# Step 4: Install Miniforge3
Write-Host "`n[4/9] Installing Miniforge3..." -ForegroundColor Yellow
$miniforgeInstaller = "$env:TEMP\Miniforge3-Windows-x86_64.exe"
$miniforgeUrl = "https://github.com/conda-forge/miniforge/releases/latest/download/Miniforge3-Windows-x86_64.exe"
$miniforgeDir = "C:\ProgramData\miniforge3"

if (!(Test-Path "$miniforgeDir\condabin\conda.bat")) {
    Write-Host "Downloading Miniforge3..." -ForegroundColor Green
    Invoke-WebRequest -Uri $miniforgeUrl -OutFile $miniforgeInstaller -UseBasicParsing

    Write-Host "Installing Miniforge3 (global install)..." -ForegroundColor Green
    Start-Process -FilePath $miniforgeInstaller -ArgumentList "/InstallationType=AllUsers", "/RegisterPython=1", "/S", "/D=$miniforgeDir" -Wait -NoNewWindow
    Remove-Item $miniforgeInstaller -Force -ErrorAction SilentlyContinue
    Write-Host "Miniforge3 installed to $miniforgeDir" -ForegroundColor Green
} else {
    Write-Host "Miniforge3 already installed at $miniforgeDir" -ForegroundColor Green
}

# Step 5: Initialize conda for PowerShell
Write-Host "`n[5/9] Initializing conda for PowerShell..." -ForegroundColor Yellow
& "$miniforgeDir\condabin\conda.bat" init powershell
Write-Host "Conda initialized for PowerShell" -ForegroundColor Green
Write-Host "NOTE: Restart PowerShell or run 'refreshenv' to activate conda" -ForegroundColor Yellow

# Step 6: Ensure OpenSSH is installed and running
Write-Host "`n[6/9] Configuring OpenSSH..." -ForegroundColor Yellow

# Check if OpenSSH Server is installed
$sshServer = Get-WindowsCapability -Online | Where-Object Name -like 'OpenSSH.Server*'
if ($sshServer.State -ne 'Installed') {
    Write-Host "Installing OpenSSH Server..." -ForegroundColor Green
    Add-WindowsCapability -Online -Name OpenSSH.Server~~~~0.0.1.0
}

# Start and enable sshd service
Start-Service sshd -ErrorAction SilentlyContinue
Set-Service -Name sshd -StartupType 'Automatic'
Write-Host "OpenSSH Server configured and running" -ForegroundColor Green

# Step 7: Configure SSH authorized keys for passwordless access
Write-Host "`n[7/9] Configuring SSH authorized keys..." -ForegroundColor Yellow

# For Administrator, use the special admin keys file
$adminKeysFile = "C:\ProgramData\ssh\administrators_authorized_keys"
$userKeysFile = "$env:USERPROFILE\.ssh\authorized_keys"

# Ensure user .ssh directory exists
$sshDir = "$env:USERPROFILE\.ssh"
if (!(Test-Path $sshDir)) {
    New-Item -ItemType Directory -Path $sshDir -Force | Out-Null
}

# Add to administrators_authorized_keys (required for Admin user)
if (Test-Path (Split-Path $adminKeysFile)) {
    if (!(Test-Path $adminKeysFile)) {
        $PublicKey | Out-File -FilePath $adminKeysFile -Encoding ascii
    }
    elseif (!(Select-String -Path $adminKeysFile -Pattern ([regex]::Escape($PublicKey.Substring(0,50))) -Quiet)) {
        Add-Content -Path $adminKeysFile -Value $PublicKey
    }
    # Fix permissions - CRITICAL for Windows OpenSSH
    icacls $adminKeysFile /inheritance:r /grant "Administrators:F" /grant "SYSTEM:F" | Out-Null
    Write-Host "Admin SSH key configured at: $adminKeysFile" -ForegroundColor Green
}

# Also add to user authorized_keys as fallback
if (!(Test-Path $userKeysFile)) {
    $PublicKey | Out-File -FilePath $userKeysFile -Encoding ascii
} elseif (!(Select-String -Path $userKeysFile -Pattern ([regex]::Escape($PublicKey.Substring(0,50))) -Quiet)) {
    Add-Content -Path $userKeysFile -Value $PublicKey
}
Write-Host "User SSH key configured at: $userKeysFile" -ForegroundColor Green

# Step 8: Restart sshd to apply changes
Write-Host "`n[8/9] Restarting SSH service..." -ForegroundColor Yellow
Restart-Service sshd
Write-Host "SSH service restarted" -ForegroundColor Green

# Step 9: Trust PSGallery for module installs
Write-Host "`n[9/9] Configuring PSGallery..." -ForegroundColor Yellow
Set-PSRepository -Name PSGallery -InstallationPolicy Trusted -ErrorAction SilentlyContinue
Write-Host "PSGallery trusted" -ForegroundColor Green

Write-Host "`n========================================" -ForegroundColor Cyan
Write-Host "Bootstrap complete!" -ForegroundColor Green
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "`nYou can now run from Linux:"
Write-Host "  ./HW_requirements/scripts/bootstrap_windows.sh $(hostname)" -ForegroundColor Gray
