#!/usr/bin/env bash
# Bootstrap a Windows machine for E2E testing from Linux
#
# Usage: ./bootstrap_windows.sh <hostname_or_ip>
#        EVE=/path/to/eve ./bootstrap_windows.sh <hostname_or_ip>
#
# PREREQUISITES:
#   1. SSH key authentication must be configured on the Windows machine.
#      Run once manually on Windows (as Administrator):
#        $key = 'ssh-rsa AAAA... your-email@example.com'
#        $key | Out-File C:\ProgramData\ssh\administrators_authorized_keys -Encoding ascii
#        icacls C:\ProgramData\ssh\administrators_authorized_keys /inheritance:r /grant 'Administrators:F' /grant 'SYSTEM:F'
#        Restart-Service sshd
#
#   2. EVE bootstrap directory must exist with wheels and DataGen.
#      See README.md for the expected directory structure.
#
# WHAT THIS SCRIPT DOES:
#   1. Creates bootstrap directory on Windows
#   2. Copies $EVE/bootstrap/* (wheels, SSH keys, DataGen) to Windows
#   3. Copies HW_requirements/scripts/*.ps1 to Windows
#   4. Runs init.ps1 (creates conda env, installs wheels)
#   5. Runs sync_*.ps1 (clones models and aie4_models repos)
#
# ENVIRONMENT VARIABLES:
#   EVE            - Base directory containing bootstrap/ (required)
#   WINDOWS_USER   - Windows username (default: Administrator)
#   WORK_DIR       - Windows working directory (default: C:\\Users\\Administrator\\Desktop\\shajaisw)
#
# EXAMPLES:
#   EVE=/path/to/eve ./bootstrap_windows.sh 10.228.203.217
#   WORK_DIR='C:\\Users\\admin\\work' EVE=/path/to/eve ./bootstrap_windows.sh w217

set -euo pipefail

# Get the directory where this script lives
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

log_info() { echo -e "${CYAN}[INFO]${NC} $1"; }
log_success() { echo -e "${GREEN}[OK]${NC} $1"; }
log_warn() { echo -e "${YELLOW}[WARN]${NC} $1"; }
log_error() { echo -e "${RED}[ERROR]${NC} $1"; }

usage() {
    echo "Usage: $0 <hostname_or_ip>"
    echo ""
    echo "Environment variables (required):"
    echo "  EVE              - Base directory containing bootstrap/ with wheels and DataGen"
    echo ""
    echo "Environment variables (optional):"
    echo "  WINDOWS_USER     - Windows username (default: Administrator)"
    echo "  WORK_DIR         - Windows working directory (default: C:\\Users\\Administrator\\Desktop\\shajaisw)"
    echo ""
    echo "See README.md for the expected EVE bootstrap directory structure."
    echo ""
    echo "Examples:"
    echo "  EVE=/path/to/eve $0 10.228.203.217"
    echo "  WORK_DIR='C:\\Users\\admin\\work' EVE=/path/to/eve $0 192.168.1.100"
    exit 1
}

# Configuration
WINDOWS_USER="${WINDOWS_USER:-Administrator}"
WORK_DIR="${WORK_DIR:-C:\\Users\\Administrator\\Desktop\\shajaisw}"
WORK_DIR_UNIX="${WORK_DIR//\\//}"  # Convert backslashes to forward slashes for scp
if [[ -z "${EVE:-}" ]]; then
    log_error "EVE environment variable is required"
    usage
fi
EVE_BOOTSTRAP="${EVE}/bootstrap"
SCRIPTS_DIR="$SCRIPT_DIR"  # HW_requirements/scripts/
BOOTSTRAP_DEST="$WORK_DIR_UNIX"
BOOTSTRAP_DEST_WIN="$WORK_DIR"  # Windows path format

# Parse arguments
if [[ $# -lt 1 ]]; then
    usage
fi

WINDOWS_HOST="$1"

# Validate prerequisites
log_info "Validating prerequisites..."

if [[ ! -d "$EVE_BOOTSTRAP" ]]; then
    log_error "EVE bootstrap directory not found: $EVE_BOOTSTRAP"
    exit 1
fi

if [[ ! -d "$SCRIPTS_DIR" ]]; then
    log_error "Scripts directory not found: $SCRIPTS_DIR"
    exit 1
fi

# Verify SSH key auth works (no password prompt)
log_info "Verifying SSH key authentication..."
if ! ssh -o BatchMode=yes -o ConnectTimeout=5 "${WINDOWS_USER}@${WINDOWS_HOST}" "echo ok" &>/dev/null; then
    log_error "SSH key auth not working for ${WINDOWS_USER}@${WINDOWS_HOST}"
    echo ""
    echo "Setup SSH key auth first by running ON the Windows machine (as Admin):"
    echo "  \$key = '$(cat ~/.ssh/id_rsa.pub)'"
    echo "  \$key | Out-File C:\\ProgramData\\ssh\\administrators_authorized_keys -Encoding ascii"
    echo "  icacls C:\\ProgramData\\ssh\\administrators_authorized_keys /inheritance:r /grant 'Administrators:F' /grant 'SYSTEM:F'"
    echo "  Restart-Service sshd"
    exit 1
fi

log_success "SSH key auth verified"

# Build SSH/SCP command helpers
ssh_cmd() {
    ssh -o StrictHostKeyChecking=no "${WINDOWS_USER}@${WINDOWS_HOST}" "$@"
}

scp_cmd() {
    scp -o StrictHostKeyChecking=no "$@"
}

# Check if rclone is available locally (faster than scp for large files)
log_info "Checking for rclone..."
if command -v rclone &>/dev/null; then
    USE_RCLONE=true
    log_success "rclone available - using rclone for faster transfers"
else
    USE_RCLONE=false
    log_warn "rclone not found locally - falling back to scp (slower for large files)"
    log_warn "Install rclone: sudo apt install rclone  OR  brew install rclone"
fi

# Copy files using rclone (sftp) or scp
copy_to_remote() {
    local src="$1"
    local dest="$2"
    local recursive="${3:-false}"

    if [[ "$USE_RCLONE" == "true" ]]; then
        # rclone copy via SFTP - much faster for many/large files
        # Format: rclone copy /local/path :sftp,host=HOST,user=USER:/remote/path
        if [[ "$recursive" == "true" ]]; then
            rclone copy "$src" ":sftp,host=${WINDOWS_HOST},user=${WINDOWS_USER}:${dest}" \
                --sftp-key-file ~/.ssh/id_rsa \
                --progress \
                --transfers 4
        else
            # For single file, use copyto
            rclone copyto "$src" ":sftp,host=${WINDOWS_HOST},user=${WINDOWS_USER}:${dest}" \
                --sftp-key-file ~/.ssh/id_rsa \
                --progress
        fi
    else
        # Fallback to scp
        if [[ "$recursive" == "true" ]]; then
            scp_cmd -r "$src" "${WINDOWS_USER}@${WINDOWS_HOST}:${dest}"
        else
            scp_cmd "$src" "${WINDOWS_USER}@${WINDOWS_HOST}:${dest}"
        fi
    fi
}

echo ""
echo "========================================"
echo "Windows Machine Bootstrap"
echo "========================================"
echo "  Host:         $WINDOWS_HOST"
echo "  User:         $WINDOWS_USER"
echo "  EVE bootstrap: $EVE_BOOTSTRAP"
echo "  Scripts:      $SCRIPTS_DIR"
echo "  Dest:         $BOOTSTRAP_DEST"
echo "========================================"
echo ""

# Step 1: Create bootstrap directory on Windows
log_info "[1/5] Creating bootstrap directory on Windows..."
ssh_cmd "powershell -Command \"New-Item -ItemType Directory -Path '$BOOTSTRAP_DEST' -Force | Out-Null\""
log_success "Bootstrap directory created"

# Step 2: Copy EVE bootstrap files (wheels, installers, etc.)
log_info "[2/5] Copying EVE bootstrap files..."
if [[ -d "$EVE_BOOTSTRAP" ]] && [[ -n "$(ls -A "$EVE_BOOTSTRAP" 2>/dev/null)" ]]; then
    copy_to_remote "${EVE_BOOTSTRAP}/" "${BOOTSTRAP_DEST}/" true
    log_success "EVE bootstrap files copied"
else
    log_warn "EVE bootstrap directory empty or not found, skipping"
fi

# Step 3: Copy PowerShell scripts from HW_requirements/scripts/
log_info "[3/5] Copying PowerShell scripts..."
for script in "${SCRIPTS_DIR}"/*.ps1; do
    if [[ -f "$script" ]]; then
        copy_to_remote "$script" "${BOOTSTRAP_DEST}/$(basename "$script")"
    fi
done
log_success "PowerShell scripts copied"

# Step 4: Run init.ps1
log_info "[4/5] Running init.ps1..."
ssh_cmd "powershell -ExecutionPolicy Bypass -File \"${BOOTSTRAP_DEST_WIN}\\init.ps1\"" || {
    log_warn "init.ps1 completed with warnings (may be expected)"
}
log_success "init.ps1 completed"

# Step 5: Run sync scripts
log_info "[5/5] Running sync scripts..."
ssh_cmd "powershell -ExecutionPolicy Bypass -File \"${BOOTSTRAP_DEST_WIN}\\sync_models_repo.ps1\""
log_success "sync_models_repo.ps1 completed"

ssh_cmd "powershell -ExecutionPolicy Bypass -File \"${BOOTSTRAP_DEST_WIN}\\sync_aie4_repo.ps1\""
log_success "sync_aie4_repo.ps1 completed"

echo ""
echo "========================================"
log_success "Bootstrap complete!"
echo "========================================"
echo ""
echo "The Windows machine is now ready for E2E testing."
echo "You can now run:"
echo "  python HW_requirements/e2e_test.py --host $WINDOWS_HOST --model psd1"
