# compile_run_qhw4_debug.ps1
# Compile and run QHW4 debug tests for a model
#
# Usage:
#   .\compile_run_qhw4_debug.ps1              # Default: psd1
#   .\compile_run_qhw4_debug.ps1 -Model psd2  # Specific model
#   .\compile_run_qhw4_debug.ps1 -Model psd1 -WorkDir "D:\test"  # Custom work dir
#   .\compile_run_qhw4_debug.ps1 -Model psd1 -SkipExecute  # Compile only

param(
    [Parameter(Position=0)]
    [string]$Model = "psd1",

    [Parameter()]
    [string]$WorkDir = "C:\Users\Administrator\Desktop\shajaisw",

    [Parameter()]
    [switch]$SkipExecute
)

$ErrorActionPreference = "Stop"

$basePath = "$WorkDir\$Model"
$compileDir = "$basePath\waic_work_compile"
$modelsSource = "$WorkDir\models\$Model"

# Validate models source exists
if (-not (Test-Path $modelsSource)) {
    Write-Error "Models source not found: $modelsSource. Run sync_models_repo.ps1 first."
    exit 1
}

# Setup base directory
New-Item -ItemType Directory -Path $basePath -Force | Out-Null
Set-Location $basePath

# Copy model files from repo
Copy-Item -Path "$modelsSource\*" -Destination . -Recurse -Force

# Clean and recreate work directory
if (Test-Path $compileDir) {
    Remove-Item -Path $compileDir -Recurse -Force -ErrorAction SilentlyContinue
}
New-Item -ItemType Directory -Path $compileDir -Force | Out-Null

# Remove old report
Remove-Item -Path "$basePath\report.csv" -Force -ErrorAction SilentlyContinue

# Activate environment and source settings
conda activate ci_etoe
. ..\settings_qhw4_pkg.ps1

# Validate required files
$modelOnnx = "$basePath\model.onnx"
$ortPath = "$basePath\DataGen\Activations\ort"
if (-not (Test-Path $modelOnnx)) {
    Write-Error "model.onnx not found: $modelOnnx"
    exit 1
}
if (-not (Test-Path $ortPath)) {
    Write-Error "ORT activations not found: $ortPath"
    exit 1
}

# Run debug script
Set-Location "C:\ProgramData\miniforge3\envs\ci_etoe\Lib\site-packages\aie4_models"

$pythonArgs = @(
    ".\debug_scripts\qhw4_op_level_debug.py",
    "-mp", "$modelOnnx",
    "-fe", "$basePath",
    "-output", "$compileDir",
    "-ort", "$ortPath\",
    "--runner", ".\debug_scripts\waic_runner.exe",
    "-wgt", "--subgraph"
)
if ($SkipExecute) {
    $pythonArgs += "--skip_running"
    Write-Host "=== Compile Only: $Model ===" -ForegroundColor Cyan
} else {
    Write-Host "=== Compile and Run: $Model ===" -ForegroundColor Cyan
}

python @pythonArgs

if ($LASTEXITCODE -ne 0) {
    Write-Error "qhw4_op_level_debug.py failed with exit code: $LASTEXITCODE"
    exit $LASTEXITCODE
}

if ($SkipExecute) {
    Write-Host "=== Compile Complete ===" -ForegroundColor Green
    Write-Host "Artifacts in: $compileDir" -ForegroundColor Yellow
} else {
    Get-Content ".\debug_scripts\report.csv"
    Copy-Item ".\debug_scripts\report.csv" "$basePath"
    Write-Host "=== Complete ===" -ForegroundColor Green
}