# Copy artifacts to remote machine using rclone
#
# Usage:
#   .\copy_to_remote.ps1 -SourcePath "C:\path\to\source" -DestHost "10.0.0.1" -DestPath "C:\path\to\dest"
#   .\copy_to_remote.ps1 -SourcePath "C:\path\to\source" -DestHost "10.0.0.1" -DestPath "C:\path\to\dest" -Username "Admin" -Password "secret"

param(
    [Parameter(Mandatory=$true)]
    [string]$SourcePath,

    [Parameter(Mandatory=$true)]
    [string]$DestHost,

    [Parameter(Mandatory=$true)]
    [string]$DestPath,

    [Parameter()]
    [string]$Username = "Administrator",

    [Parameter()]
    [string]$Password = $env:PSD_WINDOWS_PASSWORD
)

$ErrorActionPreference = "Stop"

# Validate rclone is installed
if (-not (Get-Command rclone -ErrorAction SilentlyContinue)) {
    Write-Error "rclone not found. Install with: choco install rclone"
    exit 1
}

# Validate source path exists
if (-not (Test-Path $SourcePath)) {
    Write-Error "Source path not found: $SourcePath"
    exit 1
}

# Validate password
if (-not $Password) {
    Write-Error "Password required. Set -Password or PSD_WINDOWS_PASSWORD env var"
    exit 1
}

$RemoteName = "copy_dest"

Write-Host "=== Copy Artifacts ===" -ForegroundColor Cyan
Write-Host "Source: $SourcePath"
Write-Host "Dest:   ${DestHost}:${DestPath}"
Write-Host ""

try {
    # Obscure the password
    $obscuredPass = & rclone obscure $Password 2>&1 | Out-String
    $obscuredPass = $obscuredPass.Trim()
    if ($LASTEXITCODE -ne 0) {
        Write-Error "Failed to obscure password: $obscuredPass"
        exit 1
    }

    # Create rclone remote for destination
    # Temporarily ignore stderr (rclone outputs NOTICE about missing config to stderr)
    Write-Host "Configuring rclone remote..." -ForegroundColor Yellow
    $prevErrorAction = $ErrorActionPreference
    $ErrorActionPreference = 'SilentlyContinue'
    $configOutput = & rclone config create $RemoteName sftp `
        host=$DestHost `
        user=$Username `
        pass=$obscuredPass `
        shell_type=powershell `
        disable_hashcheck=true 2>&1
    $configExitCode = $LASTEXITCODE
    $ErrorActionPreference = $prevErrorAction

    if ($configExitCode -ne 0) {
        Write-Error "Failed to create rclone remote: $configOutput"
        exit 1
    }

    # Convert destination path to SFTP format (add leading slash for drive letter)
    $sftpDestPath = $DestPath -replace '\\', '/'
    if ($sftpDestPath -match '^[A-Za-z]:') {
        $sftpDestPath = "/$sftpDestPath"
    }

    $destSpec = "${RemoteName}:${sftpDestPath}"

    Write-Host "Running rclone copy..." -ForegroundColor Yellow
    Write-Host "  From: $SourcePath"
    Write-Host "  To:   $destSpec"
    Write-Host ""

    # Run rclone copy (not sync - copy only adds files, never deletes)
    rclone copy $SourcePath $destSpec `
        --retries 5 `
        --low-level-retries 10 `
        --transfers 4 `
        --checkers 8 `
        --progress `
        --stats 10s `
        --ignore-checksum

    if ($LASTEXITCODE -ne 0) {
        Write-Error "rclone copy failed with exit code: $LASTEXITCODE"
        exit $LASTEXITCODE
    }

    Write-Host ""
    Write-Host "=== Copy Complete ===" -ForegroundColor Green

} finally {
    # Cleanup remote config
    rclone config delete $RemoteName 2>&1 | Out-Null
}
