# init.ps1
# One-time initialization script for CI environment
#
# Usage: .\init.ps1
#
# Prerequisites:
#   - Miniforge/conda installed
#   - Wheel files in current directory (from \\10.228.200.62\Users\Administrator\wheels\wheels_waic_latest\2516):
#     - waic-*.whl
#     - onnxruntime_vitisai-*.whl
#     - voe-*.whl
#   - SSH key: gitenterprise_windows
#
# NOTE: For wheels 2516+, onnxruntime/onnxruntime-extensions are removed from waic wheel.
# Installation order matters:
#   1. onnxruntime
#   2. onnxruntime-extensions
#   3. waic wheel
#   4. onnxruntime_vitisai wheel
#   5. voe wheel

$ErrorActionPreference = "Stop"

Write-Host "=== Initializing CI Environment ===" -ForegroundColor Cyan

# Configure conda to use conda-forge only (avoids Anaconda rate limits)
Write-Host "Configuring conda channels..." -ForegroundColor Yellow
$condarc = @"
channels:
  - conda-forge
channel_priority: strict
"@
# Write to user profile
$condarc | Out-File -FilePath "$env:USERPROFILE\.condarc" -Encoding utf8 -Force
# Also write to conda installation directory if it exists
$condaRoot = (conda info --base 2>$null)
if ($condaRoot -and (Test-Path $condaRoot)) {
    $condarc | Out-File -FilePath "$condaRoot\.condarc" -Encoding utf8 -Force
}

# Create conda environment (use --override-channels to ensure only conda-forge is used)
Write-Host "Creating conda environment..." -ForegroundColor Yellow
conda create -n ci_etoe python=3.12 -y --override-channels -c conda-forge
if ($LASTEXITCODE -ne 0) { throw "conda create failed" }

conda activate ci_etoe
if ($LASTEXITCODE -ne 0) { throw "conda activate failed" }

conda install gxx -y --override-channels -c conda-forge
if ($LASTEXITCODE -ne 0) { throw "conda install gxx failed" }

# Install base packages FIRST (before custom wheels)
Write-Host "Installing onnxruntime..." -ForegroundColor Yellow
pip install onnxruntime
if ($LASTEXITCODE -ne 0) { throw "pip install onnxruntime failed" }

Write-Host "Installing onnxruntime-extensions..." -ForegroundColor Yellow
pip install onnxruntime-extensions
if ($LASTEXITCODE -ne 0) { throw "pip install onnxruntime-extensions failed" }

# Install custom wheels IN ORDER
Write-Host "Installing custom wheels (order matters)..." -ForegroundColor Yellow
$wheels = @(
    "waic-*.whl",
    "onnxruntime_vitisai-*.whl",
    "voe-*.whl"
)
foreach ($pattern in $wheels) {
    $wheel = Get-ChildItem -Path . -Filter $pattern | Select-Object -First 1
    if ($wheel) {
        Write-Host "  Installing $($wheel.Name)..." -ForegroundColor Yellow
        pip install --force-reinstall $wheel.FullName
        if ($LASTEXITCODE -ne 0) { throw "Failed to install $($wheel.Name)" }
    } else {
        Write-Warning "Wheel not found: $pattern"
    }
}

pip install numpy==1.26.4
if ($LASTEXITCODE -ne 0) { throw "pip install numpy failed" }

# Remove old packages
$sitePackages = "C:\ProgramData\miniforge3\envs\ci_etoe\Lib\site-packages"
Write-Host "Cleaning old installations..." -ForegroundColor Yellow
Remove-Item "$sitePackages\dmacompiler" -Recurse -Force -ErrorAction SilentlyContinue
Remove-Item "$sitePackages\aie4_models" -Recurse -Force -ErrorAction SilentlyContinue

# Clone/sync aie4_models using sync script
Write-Host "Syncing aie4_models..." -ForegroundColor Yellow
& "$PSScriptRoot\sync_aie4_repo.ps1"
if ($LASTEXITCODE -ne 0) { throw "sync_aie4_repo.ps1 failed" }

Write-Host "=== Initialization complete ===" -ForegroundColor Green