# sync_aie4_repo.ps1
# Sync aie4_models repository to a specific commit
#
# Usage:
#   .\sync_aie4_repo.ps1                           # Sync to origin/main
#   .\sync_aie4_repo.ps1 -Commit "abc123"          # Sync to specific commit SHA
#   .\sync_aie4_repo.ps1 -Commit "my-branch"       # Sync to branch name
#   .\sync_aie4_repo.ps1 -Commit "pull/788/head"   # Sync to PR head
#   .\sync_aie4_repo.ps1 -Commit "pull/788/merge"  # Sync to PR merge commit
#   .\sync_aie4_repo.ps1 -WorkDir "D:\test"        # Custom work dir

param(
    [Parameter(Mandatory=$false)]
    [string]$Commit = "",

    [Parameter()]
    [string]$WorkDir = "C:\Users\Administrator\Desktop\shajaisw"
)

# Setup environment
. .\setup_env.ps1 -WorkDir $WorkDir

$RepoPath = "C:\ProgramData\miniforge3\envs\ci_etoe\Lib\site-packages\aie4_models"

Write-Host "=== Syncing aie4_models repository ===" -ForegroundColor Cyan
Write-Host "Repository path: $RepoPath"

# Clone if repo doesn't exist
if (-not (Test-Path "$RepoPath\.git")) {
    Write-Host "Repository not found, cloning..." -ForegroundColor Yellow
    $ParentDir = Split-Path $RepoPath -Parent
    Push-Location $ParentDir
    try {
        # Remove any partial installation
        if (Test-Path $RepoPath) {
            # Use robocopy to empty directory (handles reserved names like NUL)
            $emptyDir = "$env:TEMP\empty_dir_$(Get-Random)"
            New-Item -ItemType Directory -Path $emptyDir -Force | Out-Null
            robocopy $emptyDir $RepoPath /MIR /R:1 /W:1 | Out-Null
            Remove-Item $emptyDir -Force -ErrorAction SilentlyContinue
            Remove-Item $RepoPath -Recurse -Force -ErrorAction SilentlyContinue
        }
        git clone --recurse-submodules git@gitenterprise.xilinx.com:IPSP/aie4_models.git
        if ($LASTEXITCODE -ne 0) {
            throw "git clone failed"
        }
    } finally {
        Pop-Location
    }
}

Push-Location $RepoPath
try {
    # Fetch all updates from origin (including PR refs)
    Write-Host "Fetching from origin..." -ForegroundColor Yellow
    git fetch origin
    if ($LASTEXITCODE -ne 0) {
        throw "git fetch failed"
    }

    # Determine target ref
    if ([string]::IsNullOrWhiteSpace($Commit)) {
        $TargetRef = "origin/main"
        Write-Host "Target: $TargetRef (default)" -ForegroundColor Yellow
    } elseif ($Commit -match "^pull/(\d+)/(head|merge)$") {
        # PR ref format: pull/123/head or pull/123/merge
        $PrNumber = $Matches[1]
        $RefType = $Matches[2]
        Write-Host "Target: PR #$PrNumber (refs/pull/$PrNumber/$RefType)" -ForegroundColor Yellow
        # Fetch the PR ref explicitly (+ prefix forces update even if not fast-forward)
        git fetch origin "+pull/$PrNumber/$RefType`:pr-$PrNumber"
        if ($LASTEXITCODE -ne 0) {
            throw "Failed to fetch PR #$PrNumber"
        }
        $TargetRef = "pr-$PrNumber"
    } elseif ($Commit -match "^[a-fA-F0-9]{7,40}$") {
        # Looks like a commit SHA - try to use it directly first
        Write-Host "Target: $Commit (commit SHA)" -ForegroundColor Yellow

        # Check if commit exists locally using git rev-parse (more reliable)
        # Temporarily disable error action preference to suppress stderr
        $oldErrorActionPreference = $ErrorActionPreference
        $ErrorActionPreference = "SilentlyContinue"

        $result = git rev-parse --verify --quiet "$Commit^{commit}" 2>$null
        $commitExists = ($LASTEXITCODE -eq 0)

        if (-not $commitExists) {
            # Commit not found locally - might be a PR merge commit
            # Fetch all PR refs to try to get it
            Write-Host "Commit not found locally, fetching PR refs..." -ForegroundColor Yellow
            git fetch origin '+refs/pull/*/merge:refs/remotes/origin/pr/*/merge' 2>$null
            git fetch origin '+refs/pull/*/head:refs/remotes/origin/pr/*/head' 2>$null

            # Check again
            $result = git rev-parse --verify --quiet "$Commit^{commit}" 2>$null
            $commitExists = ($LASTEXITCODE -eq 0)
        }

        # Restore error action preference
        $ErrorActionPreference = $oldErrorActionPreference

        if (-not $commitExists) {
            throw "Commit $Commit not found. It may be from a force-pushed branch or unavailable PR."
        }
        $TargetRef = $Commit
    } else {
        # Assume it's a branch name
        $TargetRef = $Commit
        Write-Host "Target: $TargetRef (branch name)" -ForegroundColor Yellow

        # Try to fetch the branch
        git fetch origin "$Commit`:refs/remotes/origin/$Commit" 2>$null
        if ($LASTEXITCODE -eq 0) {
            $TargetRef = "origin/$Commit"
        }
    }

    # Hard reset to target
    Write-Host "Resetting to $TargetRef..." -ForegroundColor Yellow
    git reset --hard $TargetRef
    if ($LASTEXITCODE -ne 0) {
        throw "git reset failed"
    }

    # Create/update local branch
    Write-Host "Checking out branch..." -ForegroundColor Yellow
    git checkout -B main $TargetRef
    if ($LASTEXITCODE -ne 0) {
        throw "git checkout failed"
    }

    # Update submodules
    Write-Host "Updating submodules..." -ForegroundColor Yellow
    git submodule update --init --recursive
    if ($LASTEXITCODE -ne 0) {
        throw "git submodule update failed"
    }

    # Install dependencies
    Write-Host "Installing Python dependencies..." -ForegroundColor Yellow
    pip install tabulate

    # Show current state
    $CurrentCommit = git rev-parse HEAD
    $CurrentBranch = git branch --show-current
    Write-Host ""
    Write-Host "=== Sync complete ===" -ForegroundColor Green
    Write-Host "Branch: $CurrentBranch"
    Write-Host "Commit: $CurrentCommit"
    git log -1 --oneline

} catch {
    Write-Host "ERROR: $_" -ForegroundColor Red
    exit 1
} finally {
    Pop-Location
}
