# 🧪 AIE4 Hardware Test & XRT Flow — README

This repository provides a full hardware-testing workflow for AIE4 kernels, including:

- XRT execution (`xrt_flow.exe`)
- Operator/subgraph testing
- Remote or local Windows hardware execution
- Golden IO update/copy utilities
- Automated result collection (`output.json`, `output.xlsx`, summaries`)
- Optional debug logging for each test subfolder

---

## 1. XRT Flow Usage

Source code location:

```
IPSP/XRT_applications at aie4_xrt_test
```

Executable usage:

```
xrt_flow.exe <elf_file> <kernel_name> <path_bin> [N_THREAD] [N_ITER] [out_compare_dtype] [Debug_flag]
```

### Parameter Details

| Argument | Description |
|----------|-------------|
| `<elf_file>` | The compiled ELF file generated from AIE compiler flow. |
| `<kernel_name>` | Kernel name to execute — found in **config.json** used during ELF generation. |
| `<path_bin>` | Directory containing the required `.bin` files (ifm.bin, ofm.bin, param.bin, wgt.bin, control.elf, etc.). |
| `N_THREAD` *(optional)* | Number of execution threads for parallel testing. |
| `N_ITER` *(optional)* | Number of iterations for repeated test execution. |
| `out_compare_dtype` *(optional)* | Data type for output comparison (e.g., int16/int8). |
| `Debug_flag` *(optional)* | Enables debug prints from XRT application. |

---

## 2. Hardware Test Flow (`HW_test`)

Hardware test framework supports:

- Copying subfolders to the remote Windows DUT
- Running `dolphin_test_aie4.py` for each op/subgraph
- Collecting logs, results, and summaries
- Optional golden IO update/copy mode
- Operator mode (`--op`) and subgraph mode (`--sg`)
- Filtering folders via wildcard patterns
- Local Windows execution or remote execution via SSH + SMB
- Optional debug mode (detailed below)

Main entry point:

```python
from HW_requirements.aie4_test_script import HW_test
```

---

## 2.1 HW_test Full Arguments

| Argument | Description |
|----------|-------------|
| `output` | Path to output directory (Output/op_x or Output/sg_x). |
| `HW_req` | Contains test scripts (`dolphin_test_aie4.py`, `copy_golden_io.py`). |
| `host` | Remote Windows IP for execution. |
| `perf_testing` | Enables performance-only test mode. |
| `golden_io` | `copy` or `update` IO for selected subfolders. |
| `dtype` | Data type for execution (`int16`, `int8`, `uint16`, etc.). |
| `mode` | `"op"` for operator mode, `"sg"` for subgraph mode. |
| `filter_patterns` | Wildcard filter (e.g., `"*Conv*"`, `"op_Add*"`, `"MatMul"`). |
| `debug` | If `True`, adds `--print` to test command AND generates per-folder debug logs. |

---

## 3. Debug Mode (`--debug`)

When debug mode is enabled:

1. `--print` is added to the `dolphin_test_aie4.py` invocation  
2. Each operator/subgraph folder receives a debug log:

```
<output>/<subfolder>/AIE_HW_Debug.txt
```

Purpose:
- Kernel bring-up
- Hardware tracing
- Debugging AIE-level execution sequences

---

## 4. run_hw.py (Simple CLI Front-End)

You can run hardware tests directly using:

```
python run_hw.py --out_dir <Output> --host <IP> [options]
```

### Options Available

| Flag | Description |
|------|-------------|
| `--out_dir` | Output directory to test. |
| `--host` | IP address of remote DUT. |
| `--perf_testing` | Enables timing-only tests. |
| `--golden_io` | Golden IO update/copy mode. |
| `--dtype` | int16 / int8 / float16 / uint16. |
| `--filter` | Wildcard folder matcher (multiple allowed). |
| `--op` | Operator mode (default). |
| `--sg` | Subgraph mode. |
| `--debug` | Enable debug mode → generates `AIE_HW_Debug.txt` inside each folder. |

Example:

```
python run_hw.py --out_dir ./Output --host 10.228.200.87 --filter "*Conv*" --dtype int16 --debug
```

---

## 5. Running HW Tests on Existing Binaries

You can use:

```python
from common import run_hw_validation
```

### Example for `conv-test.py`:

```python
if hwtest:
    run_hw_validation(
        output_root,
        dtype="int8" if data_type == DataType.INT8 else "int16",
        filter_patterns="op_Conv*",  # optional
        debug=False,                 # True → generate logs
    )
```

Add support via:

```
--hwtest
```

---

## 6. run_hw_validation (Recommended Helper)

`run_hw_validation()` wraps `HW_test()` and supports:

- dtype selection  
- optional filtering  
- debug tracing  
- simplified invocation  

Example:

```python
run_hw_validation(
    output_root="./Output",
    dtype="int16",
    filter_patterns=None,
    debug=True,
)
```

---

## 7. Folder Structure Expectations

Operator mode (`--op`):

```
Output/
 ├── op_Add/
 │    └── hw_package/
 │         ├── control.elf
 │         ├── ifm.bin
 │         ├── ofm.bin
 │         ├── param.bin
 │         ├── wgt.bin
 ├── op_Conv/
 └── ...
```

Subgraph mode (`--sg`):

```
Output/
 ├── sg_0/
 │     ├── control.elf
 │     ├── ifm.bin
 │     ├── param.bin
 │     └── ...
 ├── sg_1/
 └── ...
```

---

## 8. Generated Outputs

After HW execution:

```
output_<timestamp>.json
output_<timestamp>.xlsx
output_summary_<timestamp>.json
```

If debug enabled:

```
<subfolder>/AIE_HW_Debug.txt
```

---

## 9. Golden IO Mode

### Copy:

```
--golden_io copy op_Add op_MatMul
```

### Update:

```
--golden_io update op_Conv op_Add
```

Automatically invokes:

```
copy_golden_io.py <subfolder> <mode>
```

---

## 10. Example Workflows

### Run all operator tests

```
python run_hw.py --out_dir ./Output --host 10.228.200.87
```

### Only run Conv operators

```
python run_hw.py --filter "*Conv*"
```

### Enable debug logging

```
python run_hw.py --debug
```

### Update golden IO

```
python run_hw.py --golden_io "update op_Conv op_Add"
```