# PyTest User Guide

This guide provides comprehensive instructions for running pytest in the `buildtest` directory.

## Prerequisites

Before running any tests, ensure you have set up your environment:

```bash
source settings.sh && source env/bin/activate
```

## Table of Contents

1. [Discovering Markers](#1-discovering-markers)
2. [Discovering Tests](#2-discovering-tests)
3. [Discovering Tests in Specific Files](#3-discovering-tests-in-specific-files)
4. [Running Tests in Specific Files](#4-running-tests-in-specific-files)
5. [Running a Specific Test in a Specific File](#5-running-a-specific-test-in-a-specific-file)
6. [Changing Target and Running Tests](#6-changing-target-and-running-tests)
7. [Submitting Tests to LSF](#7-submitting-tests-to-lsf)
8. [Automated Regression Testing and CI/CD](#8-automated-regression-testing-and-cicd)
9. [Miscellaneous Submission Patterns](#9-miscellaneous-submission-patterns)
10. [Common Test Execution Patterns](#10-common-test-execution-patterns)
11. [Environment Variables](#11-environment-variables)
12. [Hardware Testing Script](#12-hardware-testing-script)
13. [For More Information](#13-for-more-information)

---

## 1. Discovering Markers

Markers are used to categorize and filter tests. To discover all available markers:

```bash
pytest --markers
```

### Available Markers in This Project

According to `pytest.ini`, the following markers are configured:

- **`dma`**: Tests related to DMA functionality
- **`pdi`**: Tests related to PDI compilation
- **`graph`**: Tests related to graph functionality

To see which tests are marked with a specific marker (without running them):

```bash
# Discover all DMA tests
pytest -m dma --collect-only

# Discover all PDI tests
pytest -m pdi --collect-only

# Discover all graph tests
pytest -m graph --collect-only
```

---

## 2. Discovering Tests

To discover all tests in the current directory without running them:

```bash
pytest --collect-only
```

This will show you:

- All test files
- All test functions/methods
- Parameterized test variations
- Applied markers

Example output format:

```bash
<Module test_conv.py>
  <Function test_conv[l2-0]>
  <Function test_conv[l2-1]>
  ...
```

To get a more verbose listing:

```bash
pytest --collect-only -v
```

---

## 3. Discovering Tests in Specific Files

To discover tests in one or more specific test files:

```bash
# Single file
pytest buildtest/test_conv.py --collect-only

# Multiple files
pytest buildtest/test_conv.py buildtest/test_gemm.py --collect-only

# With verbose output
pytest buildtest/test_conv.py --collect-only -v
```

To discover tests in a specific file with a specific marker:

```bash
pytest buildtest/test_conv.py -m dma --collect-only
```

---

## 4. Running Tests in Specific Files

To run all tests in a specific file:

```bash
# Run all tests in test_conv.py
pytest buildtest/test_conv.py

# Run all tests in test_conv.py with verbose output
pytest buildtest/test_conv.py -v

# Run all tests in test_conv.py with extra verbose output and detailed traceback
pytest buildtest/test_conv.py -vv --tb=auto
```

To run tests from multiple files:

```bash
pytest buildtest/test_conv.py buildtest/test_gemm.py -v
```

To run tests in a file with a specific marker:

```bash
# Run only DMA tests in test_conv.py
pytest buildtest/test_conv.py -m dma -v
```

---

## 5. Running a Specific Test in a Specific File

To run a specific test function, use the format `<file>::<test_name>`:

```bash
# Run a specific test function
pytest buildtest/test_conv.py::test_conv -v

# Run a specific parameterized test variant
pytest buildtest/test_conv.py::test_conv[l2-0] -v

# Run multiple specific tests
pytest buildtest/test_conv.py::test_conv[l2-0] buildtest/test_conv.py::test_conv[l2-1] -v
```

### Understanding Test Names with Parameters

For parameterized tests, pytest generates test IDs in the format:

```bash
test_name[param1-param2-...]
```

For example:

- `test_conv[l2-0]` - Test with shape_index=0 and dataflow_type=l2
- `test_gemm_wgt_l3[int4-1]` - Test with dtype=int4 and shape_index=1
- `test_binary[add_8-0]` - Test with op=add_8 and shape_index=0

To run a specific parameterized test:

```bash
pytest buildtest/test_binary.py::test_binary[add_8-0] -v
```

---

## 6. Changing Target and Running Tests

The `buildtest/conftest.py` file defines a custom `--target` option that allows you to specify the build target for tests.

### Available Build Targets

Based on `common.py`, the following targets are available:

- **`dataflow`** (default): Standard dataflow target
- **`sim`**: Simulation target
- **`cert_sim`**: Certificate simulation target
- **`cert`**: Certificate target

### Running Tests Locally with Different Targets

You can run tests directly with pytest:

```bash
# Run with default target (dataflow)
pytest buildtest/test_conv.py -v

# Run with SIM target
pytest buildtest/test_conv.py --target sim -v

# Run with CERT target
pytest buildtest/test_conv.py --target cert -v

# Run with CERT_SIM target
pytest buildtest/test_conv.py --target cert_sim -v

# Run with custom output directory and template variables
pytest buildtest/test_conv.py --target sim --output-root "/tmp/test_{{worker_id}}_{{uuid}}" -v
```

**Template Variable Support:**

The `--output-root` option supports Jinja2 template variables when running through pytest:

- `{{worker_id}}` - Replaced with pytest-xdist worker ID (gw0, gw1, etc.) for parallel execution
- `{{uuid}}` - Replaced with a unique UUID for each test session

**Note:** Template variables are only expanded when running through pytest. If you run test scripts directly with Python (e.g., `python test_conv.py --output-root "/tmp/{{worker_id}}"`), the path will be used literally without template expansion.

### Combining Target with Markers

```bash
# Run all DMA tests with SIM target
pytest -m dma --target sim -v

# Run specific file with specific marker and target
pytest buildtest/test_conv.py -m dma --target sim -v
```

### Running Specific Tests with Custom Target

```bash
# Run specific test with SIM target
pytest buildtest/test_conv.py::test_conv[l2-0] --target sim -v

# Run multiple specific tests with CERT target
pytest buildtest/test_binary.py::test_binary[add_8-0] --target cert -v
```

---

## 7. Submitting Tests to LSF

For longer-running tests or to leverage cluster resources, use the `pytest_lsf.py` script to submit tests to the LSF job scheduler.

### Basic Usage

```bash
# Preview what would be submitted (dry-run)
python buildtest/pytest_lsf.py submit buildtest/test_binary.py::test_binary[add_8-0] --target sim --dry-run

# Actually submit the job
python buildtest/pytest_lsf.py submit buildtest/test_binary.py::test_binary[add_8-0] --target sim --submit
```

### Submitting Multiple Tests

The script automatically discovers and submits all matching tests:

```bash
# Submit all tests in a file
python buildtest/pytest_lsf.py submit buildtest/test_conv.py --target sim --submit

# Submit tests matching a marker
python buildtest/pytest_lsf.py submit -m dma --target sim --submit

# Submit tests matching a keyword
python buildtest/pytest_lsf.py submit -k "add" --target sim --submit
```

### LSF Job Configuration

Customize LSF job parameters:

```bash
# Change queue (default: medium)
python buildtest/pytest_lsf.py submit test_conv.py --target sim --queue long --submit

# Change memory limit (default: 16GB)
python buildtest/pytest_lsf.py submit test_conv.py --target sim --mem-limit 32GB --submit

# Specify output directory for logs (default: buildtest/lsf_logs)
python buildtest/pytest_lsf.py submit test_conv.py --target sim --output-dir ./my_logs --submit

# Run tests on hardware (CERT target) instead of simulation
python buildtest/pytest_lsf.py submit test_conv.py --hwtest --submit
```

### Advantages of Using LSF

- **Parallel execution**: Each test runs as a separate job, maximizing parallelism
- **Resource management**: Tests run on cluster nodes with guaranteed resources
- **Job persistence**: Jobs continue even if you disconnect
- **Output capture**: Separate stdout/stderr files for each test in `buildtest/lsf_logs/`
- **Flexible filtering**: Use all pytest selection options (-m, -k, specific tests)

### Checking Job Status

Use the built-in `status` command to check your LSF jobs:

```bash
# Show status of all running jobs
python buildtest/pytest_lsf.py status

# Show all jobs including completed
python buildtest/pytest_lsf.py status --all

# Show jobs with specific prefix
python buildtest/pytest_lsf.py status --job-prefix test_conv

# Show detailed information
python buildtest/pytest_lsf.py status -v
```

Alternatively, use LSF commands directly:

```bash
# View all your LSF jobs
bjobs

# View detailed job information
bjobs -l <job_id>

# View job output while running
bpeek <job_id>
```

### Viewing Job Logs

Use the built-in `logs` command to view job output:

```bash
# Show stdout for a job
python buildtest/pytest_lsf.py logs test_binary_add_8_0

# Show stderr
python buildtest/pytest_lsf.py logs test_binary_add_8_0 --errors

# Follow the log file (like tail -f)
python buildtest/pytest_lsf.py logs test_binary_add_8_0 --follow

# Specify custom output directory
python buildtest/pytest_lsf.py logs test_binary_add_8_0 --output-dir ./my_logs
```

Or directly view log files:

```bash
# Check stdout
cat buildtest/lsf_logs/test_binary_add_8_0.out

# Check stderr
cat buildtest/lsf_logs/test_binary_add_8_0.err

# Follow logs in real-time
tail -f buildtest/lsf_logs/test_binary_add_8_0.out
```

### Common LSF Submission Patterns

```bash
# Dry-run to preview all tests that would be submitted
python buildtest/pytest_lsf.py submit -m pdi --target sim --dry-run

# Submit all PDI tests to LSF
python buildtest/pytest_lsf.py submit -m pdi --target sim --submit

# Submit specific test with custom memory
python buildtest/pytest_lsf.py submit buildtest/test_gemm.py::test_gemm[int16-0] --target sim --mem 64GB --submit

# Submit all tests in a file to long queue
python buildtest/pytest_lsf.py submit test_conv.py --target cert --queue long --submit

# Submit hardware tests (runs on CERT target)
python buildtest/pytest_lsf.py submit -k "(test_conv or test_gemm) and ([0] or -0])" --hwtest --submit
```

---

## 8. Automated Regression Testing and CI/CD

For automated regression testing with email notifications, HTML reports, and CI/CD integration, use the scripts in the `ci/` directory.

### Quick Start

```bash
# Quick manual regression on current branch
cd ci
bash run_lsf_tests.sh -k "test_conv"

# Full automated run with email and remote branch
bash ci_cron_job.sh --remote-branch origin/main --emails "team@amd.com"

# Run hardware tests (CERT target)
bash ci_cron_job.sh --target cert --hwtest -k "(test_conv or test_gemm) and ([0] or -0])"

# Hardware tests with email notification
bash ci_cron_job.sh --target cert --hwtest --emails "team@amd.com" -k "(test_conv or test_gemm) and ([0] or -0])"

# Scheduled cron job (nightly at 2 AM)
0 2 * * * cd /path/to/aie4_models/ci && bash ci_cron_job.sh --remote-branch origin/main --emails "team@amd.com"
```

### Key Features

The CI automation system provides:

- **Automated test submission** - Submits all matching tests to LSF
- **Job monitoring** - Waits for all jobs to complete with efficient status polling
- **Failure detection** - Checks LSF exit codes and DI_FAIL errors
- **HTML/Markdown reports** - Generates detailed test reports with log excerpts
- **Email notifications** - Sends results via sendmail
- **Git branch switching** - Test local or remote branches automatically
- **Hardware testing** - Run tests on actual hardware (CERT target) with `--hwtest` flag
- **GitHub Actions integration** - Automated hardware tests on every PR with result comments
- **Performance optimizations** - Batch bjobs calls, memory-efficient log reading
- **Interrupt handling** - Gracefully handles Ctrl+C, kills running jobs

### Script Hierarchy

```bash
ci_cron_job.sh          # Automation wrapper: git, email, output management
  └─> run_lsf_tests.sh   # Environment setup: LSF, venv, dependencies
      └─> run_lsf_tests.py   # Core orchestration: submit, monitor, report
```

### When to Use What

| Script                           | Use Case                                     |
|----------------------------------|----------------------------------------------|
| `pytest_lsf.py` (section 7)      | Interactive manual testing (sim or hardware) |
| `ci/run_lsf_tests.sh`            | Quick manual regression (no email)           |
| `ci/ci_cron_job.sh`              | Full automation with email/git/cron/hardware |
| `.github/workflows/hw-tests.yml` | Automated hardware tests on every PR         |

### Complete Documentation

For comprehensive CI/CD documentation including:

- Architecture and data flow diagrams
- Complete script reference with all command-line options
- Configuration (email, git branch switching, output directories)
- Performance optimizations explained
- Integration with cron and GitHub Actions
- Troubleshooting guide

**See [ci/README.md](ci/README.md)**

---

## 9. Miscellaneous Submission Patterns

### Parallel Execution

The project is configured to run tests in parallel by default (`-n auto` in `pytest.ini`). To control parallelization:

```bash
# Run with 4 parallel workers
pytest -n 4

# Run without parallelization
pytest -n 0
```

### Filtering by Keyword

Discover tests matching a keyword expression:

```bash
# Run all tests with "conv" in the name
pytest -k conv -v --collect-only

# Run all tests with "add" in the name
pytest -k add -v --collect-only

# Run tests matching multiple keywords (OR)
pytest -k "conv or gemm" -v --collect-only

# Run tests matching pattern but excluding others
pytest -k "conv and not l3" -v --collect-only
```

Run tests matching a keyword expression:

```bash
# Run all tests with "conv" in the name
pytest -k conv -v

# Run all tests with "add" in the name
pytest -k add -v

# Run tests matching multiple keywords (OR)
pytest -k "conv or gemm" -v

# Run tests matching pattern but excluding others
pytest -k "conv and not l3" -v
```

### Show Test Output

```bash
# Show print statements during test execution
pytest -s

# Capture output but show it for failed tests only (default)
pytest --tb=auto
```

### Stop on First Failure

```bash
pytest -x  # Stop on first failure
pytest --maxfail=3  # Stop after 3 failures
```

### Rerun Failed Tests

```bash
# First run
pytest

# Rerun only failed tests from last run
pytest --lf

# Rerun failed tests first, then continue with others
pytest --ff
```

---

## 10. Common Test Execution Patterns

```bash
# Run all DMA tests in buildtest directory
pytest -m dma -v

# Run all PDI compilation tests
pytest -m pdi -v

# Run specific file with CERT target and verbose output
pytest buildtest/test_conv.py --target cert -vv

# Run specific test case with custom target
pytest buildtest/test_gemm.py::test_gemm[int16-0] --target sim -v

# Discover what tests would run with DMA marker
pytest -m dma --collect-only

# Run tests in parallel with specific target
pytest -m dma --target sim -n 4 -v

# Submit hardware tests to LSF
python buildtest/pytest_lsf.py submit -k "(test_conv or test_gemm) and ([0] or -0])" --hwtest --submit

# Run full hardware regression with email
cd ci && bash ci_cron_job.sh --target cert --hwtest --emails "team@amd.com" -k "(test_conv or test_gemm) and ([0] or -0])"
```

---

## 11. Environment Variables

Before running tests, ensure environment variables are sourced from `settings.sh`. Some tests may require specific environment variables like `LOG_ENABLED`.

---

## 12. Hardware Testing Script

For quick hardware validation without the full CI infrastructure, use the `hw_test.sh` script at the repository root:

```bash
# Run default hardware tests (conv and gemm, first shape only)
bash hw_test.sh

# Set custom hardware host IP
HW_HOST=10.18.183.200 bash hw_test.sh
```

**What it does:**

1. Creates a unique timestamped output directory
2. Runs pytest with `--target cert` for selected tests
3. Handles parallel execution with worker-specific directories
4. Flattens the directory structure after test completion
5. Executes hardware validation on the generated binaries
6. Cleans up the output directory on exit

**Configuration:**

- Tests: `(test_conv or test_gemm) and ([0] or -0])` (first shape only)
- Output: `Output_YYYYMMDD_HHMMSS` in repository root
- Hardware IP: Defaults to `10.18.183.114`, override with `HW_HOST` environment variable

**Use cases:**

- Quick smoke test on hardware before committing
- Local hardware validation without LSF infrastructure
- Debugging hardware-specific issues

---

## 13. For More Information

- Run `pytest --help` for all available options
- See `buildtest/conftest.py` for custom fixtures and command-line options
- See `pytest.ini` for project-wide pytest configuration
- Check individual test files for test-specific documentation
- See `hw_test.sh` for hardware validation script
