# AIE4 Model Benchmarks

End-to-end model benchmarks for AIE4 models. These tests run the full L2L3 allocation and compilation flow for complete models.

---

## Available Benchmarks

| Test                          | Model          | Description                                                |
|-------------------------------|----------------|------------------------------------------------------------|
| `test_resnet50`               | ResNet50 INT8  | L2 dataflows: End-to-end benchmark for ResNet50 INT8 model |
| `test_yolov3`                 | YoloV3 INT8    | L2 dataflows: End-to-end benchmark for YoloV3 INT8 model   |
| `test_ipsp_model_compilation` | IPSP models    | L3 dataflows: Nightly compilation regression (18 models)   |

---

### Prerequisites

Activate the environment:

```bash
bash
source settings.sh
source env/bin/activate
source /group/xsjfarm/lsf/conf/profile.lsf
```

---

## IPSP Model Compilation Tests

The `test_ipsp_model_compilation` tests compile models from the [IPSP/models](https://gitenterprise.xilinx.com/IPSP/models) repository. These are nightly regression tests that validate the full compilation pipeline.

### Model Repository Setup

Models are automatically cloned via sparse checkout when tests are collected. The repo is cloned to:

```text
graph/tests/ipsp_models/
```

**Automatic clone** (happens on first run):

```bash
pytest aie4_bench/test_model_compilation.py --co --run-model-compilation -n0
```

> **⚠️ Important: Automatic clone failure**
>
> Automatic clone requires git ssh key setup: Check [github-ssh-setup](https://docs.github.com/en/authentication/connecting-to-github-with-ssh/adding-a-new-ssh-key-to-your-github-account) to setup ssh-key if you have not setup ssh for your account.

**Manual clone** (if automatic fails or you need full repo):

```bash
cd graph/tests
git clone https://gitenterprise.xilinx.com/IPSP/models.git ipsp_models
```

### Model Data folder

Model data is stored at a fixed location based on model name:

```text
/everest/ppsdv_cases_nobkup/shashank/bootstrap/<model_name>/DataGen/Consts/

e.g.;

/everest/ppsdv_cases_nobkup/shashank/bootstrap/psd1/DataGen/Consts/
/everest/ppsdv_cases_nobkup/shashank/bootstrap/psd2/DataGen/Consts/
```

### Collecting Tests

List all available model compilation tests:

```bash
pytest aie4_bench/test_model_compilation.py --co --run-model-compilation -n0 -s
```

Example output:

```text
found model (compilation): 0 .../ipsp_models/psd1/model.onnx
found model (compilation): 1 .../ipsp_models/psd2/model.onnx
...
TOTAL_COMPILATION_MODELS=18
collected 18 items

<Function test_ipsp_model_compilation[get_model0]>   # psd1
<Function test_ipsp_model_compilation[get_model1]>   # psd2
...
```

### Running Specific Models

Run a single model by test ID:

```bash
# Run psd1 (get_model0)
pytest aie4_bench/test_model_compilation.py::test_ipsp_model_compilation[get_model0] \
  --run-model-compilation -n0 -sv

# Run multiple specific models
pytest aie4_bench/test_model_compilation.py \
  -k "get_model0 or get_model1" \
  --run-model-compilation -n0 -sv
```

**Using `--model-filter` (recommended for filtering by model name):**

```bash
# Filter by exact model name
pytest aie4_bench/test_model_compilation.py \
  --run-model-compilation --model-filter "psd1" -n0 -sv

# Filter with glob pattern (all psd* models)
pytest aie4_bench/test_model_compilation.py \
  --run-model-compilation --model-filter "psd*" -n0 -sv

# Filter multiple models (comma-separated)
pytest aie4_bench/test_model_compilation.py \
  --run-model-compilation --model-filter "psu0,psu1,psu2" -n0 -sv

# Combine patterns (all psd and psu models)
pytest aie4_bench/test_model_compilation.py \
  --run-model-compilation --model-filter "psd*,psu*" -n0 -sv
```

### Running All Models

```bash
pytest aie4_bench/test_model_compilation.py \
  --run-model-compilation -n0 -sv
```

### Running with Model Data

For models that have associated test data (DataGen), you can enable model data compilation:

```bash
# Only compile models that have DataGen data available
pytest aie4_bench/test_model_compilation.py \
  --run-model-compilation --with-model-data -n0 -sv
```

This will:

- Filter to only models with `data_dir` available (DataGen at `/everest/.../bootstrap/<model>/DataGen/Consts/`)
- Add `read_model_data: true` and `model_data_path` to the model config
- Enable weight/constant loading during compilation

> **⚠️ Important: Use high-performance storage for OUTPUT_DIR**
>
> NFS home directories have slow I/O and can cause compilation failures or timeouts when multiple LSF jobs write concurrently. Always use high-performance LSF-accessible storage:
>
> - `/everest/ppsdv_cases_nobkup/` (recommended)
> - Other fast scratch disks accessible from LSF nodes
>
> Do **not** use `$HOME` or NFS-mounted directories.

### Running All Models in Parallel (LSF)

For faster execution, use LSF to run all models in parallel. Each model is submitted as a separate LSF job.

**Using the helper script (recommended):**

```bash
# Default (random data)
bash aie4_bench/run_model_compilation_lsf.sh

# With model data
bash aie4_bench/run_model_compilation_lsf.sh --with-model-data

# Custom output directory
bash aie4_bench/run_model_compilation_lsf.sh --output-dir /path/to/output

# Filter specific models (supports glob patterns, comma-separated)
bash aie4_bench/run_model_compilation_lsf.sh --model-filter "psd*"
bash aie4_bench/run_model_compilation_lsf.sh --model-filter "psu0,psu1,psu2"

# Combine options
bash aie4_bench/run_model_compilation_lsf.sh --with-model-data --model-filter "psd*"

# Show help
bash aie4_bench/run_model_compilation_lsf.sh --help
```

| Option                | Description                                                                                        |
|-----------------------|----------------------------------------------------------------------------------------------------||
| `--with-model-data`   | Run compilation with model data instead of random data                                             |
| `--output-dir DIR`    | Custom output directory (default: `/everest/ppsdv_cases_nobkup/$USER/logs/model_compilation_<ts>`) |
| `--model-filter PAT`  | Comma-separated glob patterns to filter models (e.g., `psd*` or `psu0,psu1`)                       |
| `-h, --help`          | Show help message                                                                                  |

This submits 18 parallel LSF jobs and waits for completion. Results are collected in `$OUTPUT_DIR/artifacts/`.

Generate a summary report after completion:

```bash
python aie4_bench/generate_compilation_report.py --artifact-dir "$OUTPUT_DIR/artifacts"
```

### CLI Options

| Option                    | Description                                                                  |
|---------------------------|------------------------------------------------------------------------------|
| `--run-model-compilation` | **Required** to enable compilation tests (disabled by default)               |
| `--with-model-data`       | Only compile models with DataGen data; adds model data config                |
| `--model-filter <pat>`    | Comma-separated glob patterns to filter models (e.g., `psd*` or `psu0,psu1`) |
| `--artifact-dir <path>`   | Output directory for artifacts (for LSF parallel execution)                  |
| `-n0`                     | Disable xdist parallelization (use LSF instead for parallel runs)            |

### Output Artifacts

Each model generates artifacts in `aie4_bench/artifacts/<run_timestamp>/<model_name>/`:

```text
artifacts/
└── run_20251216_102208/
    └── psd1/
        ├── model_cfg.yaml      # Build configuration
        ├── build_stdout.log    # Compilation stdout
        ├── build_stderr.log    # Compilation stderr
        ├── result.json         # Pass/fail status
        └── Output/             # Compiled binaries (cleaned on success)
```

---

## Adding New Benchmarks

1. Create a new test file `test_<model>.py`
2. Add the marker:

   ```python
   @pytest.mark.e2e_model_bench
   @pytest.mark.xdist_group("serial")
   def test_<model>():
       ...
   ```

3. Follow the existing pattern in `test_yolov3.py`

---

## Model Files

Model files are stored in `graph/<model>/`:

```text
graph/
└── yolov3/
    ├── YoloV3_INT8_Model.onnx
    ├── YoloV3_INT8_Model_subgraph_cleaned_int8.onnx
    └── ...
```

IPSP models are stored in `graph/tests/ipsp_models/`:

```text
graph/tests/ipsp_models/
├── psd1/
│   ├── model.onnx
│   ├── model_fused.onnx
│   ├── model_fused_IR.json
│   ├── model_fused_IR_unique_nodes.json
│   └── tensor_map.json
├── psd2/
└── ...
```
