"""Data model for model compilation results."""
from dataclasses import dataclass, asdict
from enum import Enum
import json
from pathlib import Path


class CompilationStatus(str, Enum):
    """Status of model compilation."""
    PASSED = "PASSED"
    FAILED = "FAILED"
    UNKNOWN = "UNKNOWN"


@dataclass
class CompilationResult:
    """Result of a model compilation test."""
    model: str
    status: CompilationStatus
    subgraphs: int = 0
    error: str = ""

    def to_json(self, path: Path) -> None:
        """Write result to JSON file."""
        with path.open('w', encoding='utf-8') as f:
            json.dump(asdict(self), f, indent=2, default=str)

    @classmethod
    def from_json(cls, path: Path) -> "CompilationResult":
        """Load result from JSON file."""
        with path.open('r', encoding='utf-8') as f:
            data = json.load(f)
        return cls(
            model=data["model"],
            status=CompilationStatus(data["status"]),
            subgraphs=data.get("subgraphs", 0),
            error=data.get("error", ""),
        )
