"""Pytest configuration for aie4_bench tests

Reuses model loading infrastructure from graph/tests for IPSP model testing.
"""
import os
import socket
import subprocess
from datetime import datetime
from pathlib import Path
from fnmatch import fnmatch
import pytest
from graph.tests.test_L2L3_allocator import ModelTestData, discover_models_from_repo
from graph.utilities import logger


# Base directory for artifacts (can be overridden via environment variable)
DEFAULT_ARTIFACT_DIR = Path(__file__).parent / "artifacts"


def cleandir(directory: str | Path) -> None:
    """Clean a directory by killing processes using it and removing it."""
    directory = str(directory)
    cleanup_script = f'''
    dir="{directory}"
    if [ -d "$dir" ]; then
        lsof +D "$dir" 2>/dev/null | awk 'NR>1 {{print $2}}' | sort -u | xargs -r kill -9 2>/dev/null || true
        sleep 2
        sync
        rm -rf "$dir" 2>/dev/null || true
        if [ -d "$dir" ]; then
            mv "$dir" "${{dir}}.old.$$" 2>/dev/null || true
            rm -rf "${{dir}}.old.$$" 2>/dev/null &
        fi
    fi
    '''
    subprocess.run(["bash", "-c", cleanup_script], check=False, capture_output=True)


@pytest.fixture(scope="session")
def artifact_dir(request):
    """Session-scoped fixture that creates an artifact directory.

    For parallel LSF execution, pass --artifact-dir to specify a unique
    output directory per job. This prevents race conditions when multiple
    jobs run simultaneously on different hosts.

    If --artifact-dir is not set, uses local artifacts directory with
    timestamp + hostname + pid for uniqueness.
    """
    # Check for command line option (set by LSF job submission)
    cli_artifact_dir = request.config.getoption("--artifact-dir", default=None)

    if cli_artifact_dir:
        # Use the directory specified by command line
        session_artifact_dir = Path(cli_artifact_dir)
        logger.info("Using artifact directory from --artifact-dir: %s", session_artifact_dir)
    else:
        # Create unique directory: timestamp + hostname + pid
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        hostname = socket.gethostname()
        pid = os.getpid()
        session_artifact_dir = DEFAULT_ARTIFACT_DIR / f"run_{timestamp}_{hostname}_{pid}"
        logger.info("Using local artifact directory: %s", session_artifact_dir)

    session_artifact_dir.mkdir(parents=True, exist_ok=True)

    yield session_artifact_dir


@pytest.fixture
def get_model(request):
    """Generic fixture to get model data

    Args:
        request.param: ModelTestData with unfused_model_path and fused_model_path
    """
    if not request or not hasattr(request, 'param'):
        pytest.skip(f"Skipping request {request}")

    model_data: ModelTestData = request.param
    return model_data


def pytest_addoption(parser):
    """Add custom command line options for aie4_bench"""
    parser.addoption(
        "--run-model-compilation",
        action="store_true",
        default=False,
        help="Run model compilation tests (nightly regression, skipped by default)"
    )
    parser.addoption(
        "--with-model-data",
        action="store_true",
        default=False,
        help="Include model data (DataGen) in compilation - requires --run-model-compilation"
    )
    parser.addoption(
        "--model-filter",
        action="store",
        default=None,
        help="Filter models by name pattern (comma-separated glob patterns, e.g., 'psd*,psu*' or 'Intel_bert,psu0')"
    )
    parser.addoption(
        "--artifact-dir",
        action="store",
        default=None,
        help="Directory for test artifacts (for LSF parallel execution)"
    )


def pytest_generate_tests(metafunc):
    """Dynamically generate test parameters for IPSP models at test collection time"""
    if "get_model" not in metafunc.fixturenames:
        return

    func_name = metafunc.function.__name__

    # Handle model compilation tests (nightly regression)
    if "test_ipsp_model_compilation" in func_name:
        if not metafunc.config.getoption("--run-model-compilation", default=False):
            logger.info("Skipping test_ipsp_model_compilation tests. Use --run-model-compilation to enable.")
            return

        ipsp_models = discover_models_from_repo()
        with_model_data = metafunc.config.getoption("--with-model-data", default=False)
        model_filter = metafunc.config.getoption("--model-filter", default=None)

        # Filter models based on --with-model-data flag
        if with_model_data:
            filtered_models = [m for m in ipsp_models if m.data_dir is not None]
            logger.info("Mode: with-model-data (only models with DataGen)")
        else:
            filtered_models = ipsp_models
            logger.info("Mode: standard (all models)")

        # Apply model name filter if specified (supports comma-separated patterns)
        if model_filter:
            patterns = [p.strip().lower() for p in model_filter.split(',')]
            filtered_models = [
                m for m in filtered_models
                if any(fnmatch(m.unfused_model_path.parent.name.lower(), pattern) for pattern in patterns)
            ]
            logger.info("Model filter: %s (matched %d models)", model_filter, len(filtered_models))

        if filtered_models:
            for i, model in enumerate(filtered_models):
                logger.info("found model (compilation): %s %s", i, model.unfused_model_path)
            metafunc.parametrize("get_model", filtered_models, indirect=True)
        logger.info("TOTAL_COMPILATION_MODELS=%d", len(filtered_models))
