"""Nightly regression tests for full model compilation pipeline.

This module provides longer-running tests that exercise the complete model
compilation flow for IPSP models. These tests are intended for nightly CI runs.

Run with:
    pytest ./aie4_bench/test_model_compilation.py::test_ipsp_model_compilation --run-model-compilation -n0 -sv

Note:
- These tests pull up models from IPSP/models repository.
- Each model uses a unique output directory (inside artifact_dir) to avoid race
  conditions when running multiple models in parallel via LSF.
"""
import os
from pathlib import Path
import subprocess
import pytest
import yaml

from aie4_bench.compilation_result import CompilationResult, CompilationStatus
from aie4_bench.conftest import cleandir
from graph.tests.test_L2L3_allocator import ModelTestData
from graph.utilities import logger
from utils.subgraph_summary import summarize_noncompilable_ops

# Set AIE4_ROOT_DIR environment variable before running tests
AIE4_ROOT_DIR = Path(os.environ["AIE4_ROOT_DIR"])

# Required files in each fused_hw_package_subgraph_* directory
# Based on HW_requirements/aie4_test_script.py and dolphin_test_aie4.py
REQUIRED_SUBGRAPH_FILES = [
    "control.asm",
    "param.bin",
    "wgt.bin",
    "config.json"
]


REQUIRED_MODEL_FILES = [
    "control.elf",
    "config.json"
]


def validate_output_structure(output_dir: Path, model_name: str) -> int:
    """Validate the Output directory structure after a successful build.

    Args:
        output_dir: Path to the Output directory
        model_name: Name of the model being compiled (for error messages)

    Returns:
        Number of subgraph packages found

    Raises:
        AssertionError: If the expected structure is not found
    """
    assert output_dir.exists(), f"Output directory not found: {output_dir}"

    # Find all fused_hw_package_subgraph_* directories
    hw_package_dirs = list(output_dir.glob("fused_hw_package_subgraph_*"))
    assert hw_package_dirs, f"No fused_hw_package_subgraph_* directories found for {model_name}"

    # Validate each subgraph directory
    logger.info("Found %d subgraph packages for %s", len(hw_package_dirs), model_name)
    for hw_dir in hw_package_dirs:
        assert hw_dir.is_dir(), f"Expected directory but found file: {hw_dir}"

        for required_file in REQUIRED_SUBGRAPH_FILES:
            file_path = hw_dir / required_file
            assert file_path.exists(), f"Missing required file {required_file} in {hw_dir.name} for {model_name}"

        logger.info("Validated subgraph: %s", hw_dir.name)

    # Find Model ELF directory
    model_elf = list(output_dir.glob("model_elf"))
    assert model_elf, f"No Model ELF directory found for {model_name}"

    # Validate Model ELF directory
    for hw_dir in model_elf:
        assert hw_dir.is_dir(), f"Expected directory but found file: {hw_dir}"

        for required_file in REQUIRED_MODEL_FILES:
            file_path = hw_dir / required_file
            assert file_path.exists(), f"Missing required file {required_file} in {hw_dir.name} for {model_name}"

        logger.info("Validated Model ELF: %s", hw_dir.name)

    return len(hw_package_dirs)


def check_file(file_path: Path | None, error_message: str, result_data: CompilationResult, model_artifact_dir: Path, model_name: Path) -> None:
    """Check if a file exists, raise an error if not.

    Args:
        file_path: Path to the file to check
        error_message: Error message to raise if file is missing
    Raises:
        FileNotFoundError: If the file does not exist
    """

    if not file_path or not file_path.exists():
        result_data.status = CompilationStatus.FAILED
        result_data.error = f"{model_name.upper()} - {error_message}"
        result_data.to_json(model_artifact_dir / "result.json")
        pytest.fail(result_data.error)


@pytest.mark.e2e_model_compilation
@pytest.mark.xdist_group("serial_compilation_nightly")
def test_ipsp_model_compilation(get_model, artifact_dir, request):  # pylint: disable=redefined-outer-name
    """Nightly regression: Full model compilation pipeline for IPSP models.

    Args:
        get_model: Fixture providing model data
        artifact_dir: Session-scoped fixture providing the artifact directory
        request: Pytest request object for accessing CLI options
    """
    model_info: ModelTestData = get_model
    model_name = model_info.unfused_model_path.parent.name
    with_model_data = request.config.getoption("--with-model-data", default=False)

    mode_str = "with-model-data" if with_model_data else "standard"
    logger.info("Compiling model: %s (mode: %s)", model_name, mode_str)

    # Create per-model artifact directory
    model_artifact_dir = artifact_dir / model_name
    model_artifact_dir.mkdir(parents=True, exist_ok=True)

    # Use unique output directory per model inside artifact_dir
    output_dir = model_artifact_dir / "Output"
    output_dir.mkdir(parents=True, exist_ok=True)

    # Prepare result using dataclass
    result_data = CompilationResult(
        model=model_name,
        status=CompilationStatus.UNKNOWN,
    )

    # check if all model files exist
    check_file(model_info.fused_ir_json, "Fused IR JSON path does not exist", result_data, model_artifact_dir, model_name)
    check_file(model_info.fused_ir_unique_nodes_json, "Fused IR unique nodes JSON path does not exist", result_data, model_artifact_dir, model_name)
    # tensor map check removed since PSI model does not have tensor map
    # check_file(model_info.tensor_map_json, "Tensor map JSON path does not exist", result_data, model_artifact_dir, model_name)

    # Write model config
    model_cfg_path = model_artifact_dir / "model_cfg.yaml"
    model_cfg = {
        "fused_model": str(model_info.fused_model_path.resolve()),
        "ir_json": str(model_info.fused_ir_json.resolve()),
        "unfused_model": str(model_info.unfused_model_path.resolve()),
        "unique_nodes": str(model_info.fused_ir_unique_nodes_json.resolve()),
        "outfolder": str(output_dir.resolve()),
        "target": "cert",
        "clean": False,
        "build_pdi": False,
    }
    if model_info.tensor_map_json:
        model_cfg["tensor_map"] = str(model_info.tensor_map_json.resolve())

    # Add model data config if --with-model-data flag is set
    if with_model_data and model_info.data_dir:
        model_cfg["read_model_data"] = True
        model_cfg["model_data_path"] = str(model_info.data_dir.resolve())
        logger.info("Using model data from: %s", model_info.data_dir)

    with model_cfg_path.open('w', encoding='utf-8') as cfg_file:
        yaml.dump(model_cfg, cfg_file, default_flow_style=False)

    # Output file paths
    stdout_path = model_artifact_dir / "build_stdout.log"
    stderr_path = model_artifact_dir / "build_stderr.log"

    script_path = AIE4_ROOT_DIR / "build_aie4.py"
    cleandir(output_dir)

    try:
        with stdout_path.open('w', encoding='utf-8') as stdout_file, \
             stderr_path.open('w', encoding='utf-8') as stderr_file:
            result = subprocess.run(
                ["python", str(script_path), "--cfg", str(model_cfg_path.absolute())],
                cwd=AIE4_ROOT_DIR,
                stdout=stdout_file,
                stderr=stderr_file,
                text=True,
                check=False,
            )

        if result.returncode != 0:
            result_data.status = CompilationStatus.FAILED
            result_data.error = f"Compilation failed {model_name}. Logs: {model_artifact_dir}"
            result_data.to_json(model_artifact_dir / "result.json")
            logger.error("Build failed. See logs at: %s", model_artifact_dir)
            cleandir(output_dir)
            pytest.fail(result_data.error)

        # Validate output structure before cleanup
        subgraph_count = validate_output_structure(output_dir, model_name)
        result_data.status = CompilationStatus.PASSED
        result_data.subgraphs = subgraph_count

        # Subgraph Count Summary
        subgraph_summary_path = model_artifact_dir / "subgraph_summary.log"
        alloc_json_path = output_dir / "model_mod_nhwc_fused.onnx_alloc.json"

        try:
            summary = summarize_noncompilable_ops(alloc_json_path)
        except Exception:  # pylint: disable=broad-except
            summary = "Subgraph Summary Generation failed!"

        subgraph_summary_path.write_text(
            f"## Subgraph Summary: `{model_name}`\n\n"
            f"{summary}\n\n"
            "---\n\n"
        )

        # Write result JSON
        result_data.to_json(model_artifact_dir / "result.json")

        # Clean up output directory on success (it's large and not needed after validation)
        cleandir(output_dir)

        logger.info("MODEL_COMPILATION_PASSED: %s", model_name)
        logger.info("Artifacts: %s", model_artifact_dir)

    except (KeyboardInterrupt, SystemExit) as e:
        # Handle timeout/kill - write result before exiting
        result_data.status = CompilationStatus.FAILED
        result_data.error = f"Compilation interrupted (timeout/killed) for {model_name}"
        result_data.to_json(model_artifact_dir / "result.json")
        logger.error("Build interrupted: %s", model_name)
        raise e
