"""
Automates building and updating the `aiebu-asm` binary for aie4_models.

This script performs the following:
1. Creates a temporary folder <REPO_ROOT>/temp.
2. Clones the 'main-ge' branch of https://github.com/Xilinx/aiebu
   into <REPO_ROOT>/temp/aiebu (with submodules).
3. Builds aiebu depending on the provided argument:

   Linux:
     - cd temp/aiebu/build
     - rm -rf Debug
     - source /tool/pandora64/etc/modules/INIT/bash
     - module load boost
     - export LDFLAGS="$LDFLAGS -ldl"
     - ./build.sh
     - If temp/aiebu/build/Debug/opt/xilinx/aiebu/bin/aiebu-asm exists,
       copy it to <REPO_ROOT>/cert_sim/
     - Write <REPO_ROOT>/cert_sim/aiebu_linux_commit.txt with the clone commit hash

   Windows:
     - cd temp\\aiebu\\build
     - Delete WBuild
     - .\\build22.bat -opt
     - If temp\\aiebu\\build\\WBuild\\Release\\xilinx\\aiebu\\aiebu-asm.exe exists,
       copy it to <REPO_ROOT>\\prebuilt\\
     - Write <REPO_ROOT>\\prebuilt\\aiebu_windows_commit.txt with the clone commit hash

4. Deletes <REPO_ROOT>/temp.
5. Stages and commits the updated files.

Usage:
    python update_aiebu.py linux
    python update_aiebu.py windows
"""

import argparse
import os
import shutil
import stat
import subprocess
import sys

AIEBU_URL = "https://github.com/Xilinx/aiebu"
AIEBU_BRANCH = "main-ge"


def run(cmd, cwd=None):
    """
    Execute a command and raise an exception if it fails.
    """
    print(f"[RUN] {' '.join(cmd)} (cwd={cwd or os.getcwd()})")
    subprocess.run(cmd, cwd=cwd, check=True)


def run_allow_fail(cmd, cwd=None):
    """
    Execute a command but do NOT raise if it fails.
    Used for build steps where tests may fail but artifacts still exist.
    """
    print(f"[RUN (allow-fail)] {' '.join(cmd)} (cwd={cwd or os.getcwd()})")
    proc = subprocess.run(cmd, cwd=cwd)
    if proc.returncode != 0:
        print(
            f"\033[93m[WARN] Command exited with code {proc.returncode}, "
            f"continuing because artifacts may still be present.\033[0m"
        )
    return proc.returncode


def get_repo_root():
    """
    Returns the directory where this script resides.
    """
    return os.path.dirname(os.path.abspath(__file__))


def _rmtree_force(path: str) -> None:
    """Robustly remove a directory tree."""

    def _onerror(func, p, exc_info):
        exc = exc_info[1]
        if isinstance(exc, PermissionError):
            try:
                os.chmod(p, stat.S_IWRITE)
                func(p)
            except PermissionError:
                print(f"\033[93m[WARN] Could not delete {p} (still in use).\033[0m")
        else:
            raise exc

    if os.path.exists(path):
        shutil.rmtree(path, onerror=_onerror)


def clone_aiebu(temp_dir: str) -> str:
    """
    Clone the aiebu repository (main-ge branch, with submodules) into temp_dir/aiebu.
    """
    clone_dir = os.path.join(temp_dir, "aiebu")
    if os.path.exists(clone_dir):
        _rmtree_force(clone_dir)

    run(
        [
            "git",
            "clone",
            "--branch",
            AIEBU_BRANCH,
            "--single-branch",
            "--recurse-submodules",
            AIEBU_URL,
            clone_dir,
        ]
    )

    return clone_dir


def get_latest_commit_hash(repo_dir: str) -> str:
    """
    Get short hash of HEAD commit inside a cloned repo.
    """
    out = subprocess.run(
        ["git", "rev-parse", "--short", "HEAD"],
        cwd=repo_dir,
        capture_output=True,
        text=True,
        check=True,
    )
    return out.stdout.strip()


def build_linux(cloned_dir: str) -> str:
    """
    Linux build:

      cd temp/aiebu/build
      rm -rf Debug
      source /tool/pandora64/etc/modules/INIT/bash
      module load boost
      export LDFLAGS="$LDFLAGS -ldl"
      ./build.sh

    Returns the expected path to the built aiebu-asm binary.
    """
    build_dir = os.path.join(cloned_dir, "build")
    os.makedirs(build_dir, exist_ok=True)

    # Remove old Debug directory if present
    debug_dir = os.path.join(build_dir, "Debug")
    if os.path.exists(debug_dir):
        _rmtree_force(debug_dir)

    # Run the build in a bash login shell so that `module` works.
    bash_cmd = (
        "source /tool/pandora64/etc/modules/INIT/bash && "
        "module load boost && "
        'export LDFLAGS="$LDFLAGS -ldl" && '
        "./build.sh"
    )
    run_allow_fail(["bash", "-lc", bash_cmd], cwd=build_dir)

    # Expected Linux binary path
    bin_path = os.path.join(
        build_dir, "Debug", "opt", "xilinx", "aiebu", "bin", "aiebu-asm"
    )
    return bin_path


def build_windows(cloned_dir: str) -> str:
    """
    Windows build:

      cd temp\\aiebu\\build
      Delete WBuild
      .\\build22.bat -opt

    Returns the expected path to the built aiebu-asm.exe binary.
    """
    build_dir = os.path.join(cloned_dir, "build")
    os.makedirs(build_dir, exist_ok=True)

    wbuild_dir = os.path.join(build_dir, "WBuild")
    if os.path.exists(wbuild_dir):
        _rmtree_force(wbuild_dir)

    # Run build22.bat, allowing it to fail (tests might fail).
    run_allow_fail(["cmd", "/c", "build22.bat", "-opt"], cwd=build_dir)

    # Expected Windows binary path
    bin_path = os.path.join(
        build_dir,
        "WBuild",
        "Release",
        "xilinx",
        "aiebu",
        "aiebu-asm.exe",
    )
    return bin_path


def copy_binary_if_exists(bin_path: str, dst_dir: str, binary_name: str) -> bool:
    """
    If bin_path exists, copy it into dst_dir with the given binary_name.
    Returns True if copied, False otherwise.
    """
    if not os.path.isfile(bin_path):
        print(
            f"\033[91m[ERROR] Expected binary not found at: {bin_path}\033[0m"
        )
        return False

    os.makedirs(dst_dir, exist_ok=True)
    dst_path = os.path.join(dst_dir, binary_name)
    print(f"[INFO] Copying {bin_path} -> {dst_path}")
    shutil.copy2(bin_path, dst_path)
    return True


def write_commit_file(target_dir: str, filename: str, commit_hash: str) -> str:
    """
    Write the commit hash to target_dir/filename.
    Returns the full path to the file.
    """
    os.makedirs(target_dir, exist_ok=True)
    path = os.path.join(target_dir, filename)
    with open(path, "w", encoding="utf-8") as f:
        f.write(commit_hash + "\n")
    print(f"[INFO] Wrote commit hash {commit_hash} to {path}")
    return path


def git_stage_and_commit(repo_root: str, paths_to_stage, target: str) -> None:
    """
    Stage given paths and create a commit with a target-specific message.
    """
    msg = (
        "Update aiebu-asm Linux build"
        if target == "linux"
        else "Update aiebu-asm Windows build"
    )

    paths_to_stage = list(paths_to_stage)

    # Verify each file exists before staging
    missing = []
    for relpath in paths_to_stage:
        abspath = os.path.join(repo_root, relpath)
        if not os.path.exists(abspath):
            missing.append(relpath)

    if missing:
        raise RuntimeError(
            f"Cannot stage missing files: {missing}. "
            "Build may have failed; aborting."
        )

    print(f"[INFO] Staging only these paths: {paths_to_stage}")


    run(["git", "add"] + paths_to_stage, cwd=repo_root)
    run(["git", "commit", "-m", msg], cwd=repo_root)

    print(f"[INFO] Commit created: {msg}")


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("target", choices=["linux", "windows"])
    args = parser.parse_args()
    target = args.target

    repo_root = get_repo_root()
    temp_dir = os.path.join(repo_root, "temp")

    if os.path.exists(temp_dir):
        _rmtree_force(temp_dir)
    os.makedirs(temp_dir, exist_ok=True)

    try:
        # 1) Clone
        cloned = clone_aiebu(temp_dir)
        commit_hash = get_latest_commit_hash(cloned)
        print(f"[INFO] Cloned aiebu @ {commit_hash}")

        staged_paths = []

        if target == "linux":
            # 2) Build for Linux
            bin_path = build_linux(cloned)
            cert_sim_dir = os.path.join(repo_root, "cert_sim")

            if not copy_binary_if_exists(
                bin_path, cert_sim_dir, "aiebu-asm"
            ):
                raise RuntimeError(
                    "Linux build did not produce aiebu-asm; aborting."
                )

            # 3) Write commit file
            commit_file = write_commit_file(
                cert_sim_dir, "aiebu_linux_commit.txt", commit_hash
            )

            staged_paths.extend(
                [
                    os.path.relpath(cert_sim_dir, repo_root),
                    os.path.relpath(commit_file, repo_root),
                ]
            )

        else:  # windows
            # 2) Build for Windows
            bin_path = build_windows(cloned)
            prebuilt_dir = os.path.join(repo_root, "prebuilt")

            if not copy_binary_if_exists(
                bin_path, prebuilt_dir, "aiebu-asm.exe"
            ):
                raise RuntimeError(
                    "Windows build did not produce aiebu-asm.exe; aborting."
                )

            # 3) Write commit file
            commit_file = write_commit_file(
                prebuilt_dir, "aiebu_windows_commit.txt", commit_hash
            )

            staged_paths.extend(
                [
                    os.path.relpath(prebuilt_dir, repo_root),
                    os.path.relpath(commit_file, repo_root),
                ]
            )

        # 4) Remove temp dir
        _rmtree_force(temp_dir)

        # 5) Stage and commit
        git_stage_and_commit(repo_root, staged_paths, target)

        print("[INFO] Done.")
        print(
            "\033[92m[SUCCESS] Changes have been staged and committed. "
            "You only need to push now.\033[0m"
        )

    except Exception as e:
        print(f"\033[91m[ERROR] {e}\033[0m", file=sys.stderr)
        if os.path.exists(temp_dir):
            _rmtree_force(temp_dir)
        sys.exit(1)


if __name__ == "__main__":
    main()
