# Registering a New Operator & Kernel Selector

This guide explains how to register a new operator in `OperatorsRegistry` and optionally attach a `kernel_selector` to control which kernel metadata is chosen during PDI generation.


## 1. Registering a New Operator

To register a new operator, call:

```python
OperatorsRegistry.add_operator(
    name_or_list_of_names,
    metadata_dict,
    group_key=OPTIONAL_GROUP_KEY
)
```

### **Required metadata fields**

Every operator must provide:

```python
"testbench": list[str]
"dataflow_script": str
"build_script": str
"kernel_names": dict[str, int] OR list[str]
"kernel_includes": list[str]
```

### **Optional fields**

Operators may also define variant kernel metadata:

```
kernel_names_<suffix>
kernel_includes_<suffix>
```

Examples:

```python
kernel_names_dwc
kernel_includes_dwc

kernel_names_1
kernel_includes_1
```

These allow the selector to switch between variants.

### Example: Simple operator registration

```python
OperatorsRegistry.add_operator(
    "Add",
    {
        "testbench": ["add.cpp"],
        "dataflow_script": "add_schedule.py",
        "build_script": "build_add.py",
        "kernel_names": {"run_add": get_kernel_id("run_add")},
        "kernel_includes": ["add/add.cc"]
    }
)
```

## 2. Adding a Kernel Selector (Optional)

If the operator needs multiple kernel variants depending on model attributes, create a selector by subclassing `BaseKernelSelector`.

### Step A: Create a selector class

```python
class MyOpKernelSelector(BaseKernelSelector):
    def select(self, field, attrs, operator, metadata):
        # Example rule
        if attrs.get("use_variant") and f"{field}_variant" in metadata:
            return f"{field}_variant"
        return field
```

### Step B: Register the operator with the selector

```python
OperatorsRegistry.add_operator(
    "MyOp",
    {
        "testbench": [...],
        "dataflow_script": "myop_schedule.py",
        "build_script": "build_myop.py",

        "kernel_names": {...},
        "kernel_includes": [...],

        "kernel_names_variant": {...},   # optional variant
        "kernel_includes_variant": [...],

        "kernel_selector": MyOpKernelSelector(),   # attach selector
    }
)
```


## 3. How Kernel Selection Works During PDI Generation

Inside the PDI generator:

```python
attrs, operator = parse_json_to_dict_with_op(block)

kernel_names = OperatorsRegistry.get_kernel_names(operator, attrs)
kernel_includes = OperatorsRegistry.get_kernel_includes(operator, attrs)
```

* `attrs` = dictionary of JSON attributes for that operator instance
* The attached selector decides which metadata key to return
* If no selector is present → base metadata (`kernel_names`, `kernel_includes`) is used


## 4. How to Add Multiple Kernel Selection Rules

If your operator has multiple variants, just define more metadata fields:

```python
kernel_names_pw
kernel_includes_pw
kernel_names_large
kernel_includes_large
```

And implement the logic in your selector:

```python
class ConvKernelSelector(BaseKernelSelector):
    def select(self, field, attrs, operator, metadata):
        if attrs.get("is_dwc") and f"{field}_dwc" in metadata:
            return f"{field}_dwc"
        if attrs.get("is_pointwise") and f"{field}_pw" in metadata:
            return f"{field}_pw"
        return field
```

---

# Implementing a New Operator With `OpBuild`

`OpBuild` is the abstract base class for all operator build flows in the AIE-4 pipeline.
Every new operator must subclass `OpBuild` and implement a small set of required methods that describe:

* How to parse its JSON attributes
* How to compute shapes
* How to run the tiler
* How to build L2 and L3 schedules
* How to emit preproc metadata
* What kernels and includes it uses by default

This provides a unified end-to-end “operator compiler” interface.


# 1. Create Your Operator Class

Each operator must define a new class that inherits from `OpBuild`.
Example:

```python
class MyAddOpBuild(OpBuild):
    ...
```


# 2. Implement Required Abstract Methods

Your class must implement **all abstract methods** from `OpBuild`:

### **(A) default_kernel_names()**

Return the base kernel names for this operator.

```python
def default_kernel_names(self) -> Dict[str, int]:
    return {"run_add": get_kernel_id("run_add")}
```

### **(B) default_kernel_includes()**

Return source files needed to build the kernels.

```python
def default_kernel_includes(self) -> List[str]:
    return ["add/add.cc"]
```

### **(C) op_type()**

Return the dataclass describing this operator’s structure.

```python
def op_type(self) -> Type[AddOp]:
    return AddOp
```

### **(D) _parse_from_dict()**

Convert the ONNX/JSON operator dictionary into your operator dataclass.

```python
def _parse_from_dict(self, data, shim_prm_offset, shim_wgt_offset, read_bins, read_model_data, model_data_path):
    return AddOp(
        input_shape=data["inputs"][0]["shape"],
        output_shape=data["outputs"][0]["shape"],
    )
```

### **(E) shape()**

Return whatever the tiler needs (often a shape object).

```python
def shape(self, op_class: AddOp):
    return op_class.input_shape
```

### **(F) tiler()**

Return the initial `ScheduleInputs` produced by the tiler.

```python
def tiler(self, dims_shape, op_class):
    return run_add_tiler(dims_shape)
```

### **(G) L2_schedule() and L3_schedule()**

Implement L2 and L3 scheduling flows.

```python
def L2_schedule(self, schedule_input):
    return add_schedule_l2(schedule_input)

def L3_schedule(self, schedule_input):
    return add_schedule_l3(schedule_input)
```

### **(H) preproc()**

Emit any preproc JSON metadata.

```python
def preproc(self, schedule_input, op_class):
    save_add_preproc(schedule_input)
```


# 3. Running the Operator Build

Once implemented, your operator becomes callable from any part of the toolchain via:

```python
next_prm, next_wgt = my_op_build.run_op(json_dict, backend=BackEnd.Adf)
```

`run_op()` automatically performs:

1. JSON → dataclass parsing
2. Shape computation
3. Tiler execution
4. L2 or L3 flow execution
5. Preproc generation
6. Producing next parameter/wgt offsets

You do **not** need to manually call scheduling functions.


# 4. Using Custom Kernel Names or Includes (Optional)

If an operator has variants (e.g., DWC vs normal conv):

1. Add multiple fields:

```
kernel_names_variant
kernel_includes_variant
kernel_names_other
kernel_includes_other
```

2. Attach a kernel selector (in OperatorRegistry, not OpBuild):

```python
"kernel_selector": MyConvKernelSelector()
```

3. `OpBuild.run_op()` will automatically receive the correct `kernel_names` and `kernel_includes` from the registry and pass them into scheduling.


# 5. Summary Checklist for Adding a New Operator

* [ ] Create a new `OpBuild` subclass
* [ ] Implement all abstract methods
* [ ] Create a dataclass for operator attributes
* [ ] Define default kernels and includes
* [ ] Implement tiler and schedule logic
* [ ] Register operator in `OperatorsRegistry`
* [ ] (Optional) Add a kernel selector for multiple variants

Once these steps are done, your operator is fully integrated into the AIE4 build system.

---
