'''
Pytest configuration for buildtest
'''
import os
import argparse
import uuid
import pytest
from jinja2 import Template
from common import BuildTarget


def _parse_target(value: str) -> BuildTarget:
    '''Parse target string to BuildTarget enum'''
    try:
        return BuildTarget[value.upper()]
    except KeyError as exc:
        valid_targets = [t.name.lower() for t in BuildTarget]
        raise argparse.ArgumentTypeError(
            f"Invalid target '{value}'. Valid options: {', '.join(valid_targets)}"
        ) from exc


def pytest_addoption(parser):
    '''Add custom command line options'''
    parser.addoption(
        "--target",
        action="store",
        default=BuildTarget.DATAFLOW,
        type=_parse_target,
        help=f"Build target for the operator. Valid options: {', '.join(t.name for t in BuildTarget)}"
    )
    parser.addoption(
        "--hwtest",
        action="store_true",
        default=False,
        help="Run HW_test after builds"
    )
    parser.addoption(
        "--clean",
        action="store_true",
        default=False,
        help="Clean output directory before running tests"
    )
    parser.addoption(
        "--output-root",
        default=os.path.join(os.path.dirname(os.path.abspath(__file__)), "..", "Output"),
        help="Output root directory"
    )


@pytest.fixture(scope="session")
def target(request):
    '''Fixture to get the build target from command line'''
    return request.config.getoption("--target")


@pytest.fixture(scope="session")
def hwtest(request):
    '''Fixture to get the hwtest flag from command line'''
    return request.config.getoption("--hwtest")


@pytest.fixture(scope="session")
def clean(request):
    '''Fixture to get the clean flag from command line'''
    return request.config.getoption("--clean")


@pytest.fixture(scope="session")
def output_root(request):
    '''Fixture to get the output root directory from command line'''
    base_output = request.config.getoption("--output-root")

    # Get xdist worker ID if running in parallel
    worker_id = os.environ.get("PYTEST_XDIST_WORKER", "master")

    # Generate a unique UUID for this session
    unique_id = str(uuid.uuid4())

    # Use Jinja2 to render template - supports {{worker_id}} and {{uuid}}
    template = Template(base_output)
    return template.render(worker_id=worker_id, uuid=unique_id)
