'''
Regression test for the binary layers
'''
import os
from typing import Optional
import typer
import pytest

from common import (
    process_simulation_results,
    create_hw_package,
    write_csv,
    default_row_mapper,
    change_dir,
    BuildTarget,
    run_hw_validation,
    clean_output_dir,
    Counter
)
from build_aie4 import compile_operator, out_dir_name_from_dict

CURRDIR = os.path.dirname(os.path.abspath(__file__))
pdi_counter = Counter()


NNI_SHAPES = [
    # Ni, Yi, Xi, Ci, No, Yo, Xo, Co, N_interpolation
    # PSD5
    [1, 16, 16, 1280, 1, 32, 32, 1280, 2],
    [1, 32, 32, 640, 1, 64, 64, 640, 2],
    # PSR
    [1, 8, 8, 1280, 1, 16, 16, 1280, 2],
]

extra_table = {
    "input_addr": {0: 1*(2**20)},
    "output_addr": {2: 1*(2**20)},
    "wgt_addr": [[0, [(0 + (0*128*1024)), (0 + (1*128*1024))]],
                 [1, [(0 + (0*128*1024)), (0 + (1*128*1024))]],
                 [2, [(0 + (0*128*1024)), (0 + (1*128*1024))]]],
    "prm_addr": [[0, (0 + (4*128*1024))], [0, (0 + (4*128*1024) + 4096)], [2, (0 + (2*128*1024))]],
    "load_input_from_ddr": True, "store_output_to_ddr": True,
    "L3": {
        "ifm": [1, 0],
        "ofm": [0, 0]
        },
    "attributes": {
        "scales_1": [1],
        "scales_2": [0],
        "scales_3": [0],
        "scales_4": [1],
    }
}


@pytest.mark.dma
@pytest.mark.parametrize("shape_index", range(len(NNI_SHAPES)))
def test_resize(shape_index: int, target: BuildTarget,  hwtest: bool, clean: bool, output_root: str) -> None:
    """Test Resize operator"""
    main(target=target, shape_index=shape_index, hwtest=hwtest, clean=clean, output_root=output_root)


def main(
    target: BuildTarget = typer.Option(default=BuildTarget.DATAFLOW, help="Build target for the operator"),
    shape_index: Optional[int] = typer.Option(default=None, help="Index of the shape to test"),
    vcd: bool = typer.Option(default=False, help="Dump VCD trace", is_flag=True),
    hwtest: bool = typer.Option(default=False, help="Run HW_test after builds", is_flag=True),
    clean: bool = typer.Option(default=False, help="Clean output directory before running", is_flag=True),
    output_root: str = typer.Option(default=os.path.join(CURRDIR, "..", "Output"), help="Root directory for output")
) -> None:
    '''Function for running resize regression testing using build script'''
    if shape_index is not None:
        shape_table = [NNI_SHAPES[shape_index]]
    else:
        shape_table = NNI_SHAPES

    is_cert_backend = target == BuildTarget.CERT
    gen_pdi = is_cert_backend and pdi_counter() == 1
    # Initialize parameters
    results_list = [''] * len(shape_table)
    simtime_list = [0.0] * len(shape_table)
    output_root = str(output_root)
    clean_output_dir(output_root, clean)
    with change_dir("../"):
        operator_name = "Resize_qdq_uint16xuint16"
        for shape in shape_table:
            extra_table["attributes"]["scales_2"] = [shape[-1]]
            extra_table["attributes"]["scales_3"] = [shape[-1]]
            merged_shape = {
                "input": shape[:4],
                "output": shape[4:8],
                "op": operator_name,
                **extra_table
                }
            print(f"Building NNI with shape: {shape}")
            os.environ["LOG_ENABLED"] = "true"
            compile_operator(operator_name,
                             merged_shape,
                             target,
                             output_root,
                             gen_standalone_pdi=gen_pdi,
                             gen_op_elf=is_cert_backend,
                             dump_vcd=vcd,
                             )
            build_dir = os.path.join(output_root, f"op_{operator_name}_shape_{out_dir_name_from_dict(merged_shape)}")
            if target == BuildTarget.SIM:
                sim_log = os.path.join(build_dir, 'AIESimulator.log')
                process_simulation_results(sim_log, shape_table.index(shape),
                                           results_list, simtime_list)
            elif target == BuildTarget.CERT:
                create_hw_package(build_dir)
            gen_pdi = False

    # Write results to CSV
    if target == BuildTarget.SIM:
        fieldnames = ['Ni', 'Yi', 'Xi', 'Ci', 'No', 'Yo', 'Xo', 'Co', 'N_interpolation']
        csv_file_name = os.path.join(CURRDIR, 'nni_aiesim_results.csv')
        write_csv(shape_table, results_list,
                  simtime_list, csv_file_name,
                  fieldnames, default_row_mapper(fieldnames))

    if hwtest:
        run_hw_validation(
            output_root,
            dtype="int16",
            debug=False,
        )


if __name__ == '__main__':
    typer.run(main)
