# Running CERT Sim for a Subgraph

This document describes the end-to-end steps required to run **CERT Sim** for a **single subgraph**, including required build changes, host-side configuration, and simulation execution.


## Overview

The flow consists of:

1. Restricting the build to the operators in the target subgraph
2. Generating CERT Sim artifacts via `test_pdi.py`
3. Configuring `host/pdi.cpp` with correct buffer sizes and binaries
4. Running CERT Sim using the subgraph `control.elf` and `param.bin`


## Step-by-Step Instructions


### 1. Restrict build to subgraph operators

Add **all operators included in the subgraph** to the global `LIST_OPS` variable in:

```
buildtest/test_pdi.py
```

This ensures that all the intended kernels in subgraph is compiled and packaged for CERT Sim.


### 2. Generate CERT Sim artifacts

```bash
cd buildtest
python test_pdi.py --target cert_sim --vcd
```

### 3. Configure host-side driver (`host/pdi.cpp`)

Edit the generated file:

```
host/pdi.cpp
```

#### 3.1 Set parameter size

Update the number of operators in the subgraph and parameter size:

```cpp
int NUM_OPS_IN_SUBGRAPH = <K>;
std::size_t prm_size = 12288 * NUM_OPS_IN_SUBGRAPH;
```


#### 3.2 Set buffer sizes (in bytes)

Update the following fields to **exactly match the binary sizes**:

```cpp
int ifm_size     = <bytes in ifm.bin>;
int wgt_size     = <bytes in wgt.bin>;
int aie_ofm_size = <bytes produced by the subgraph>;
int cpu_ofm_size = <bytes in cpu_ofm.bin>; // optional if not comparing
```

Incorrect sizes will lead to partial reads, garbage data, or simulation failures.


#### 3.3 Provide binary paths

Uncomment and update the binary load paths:

```cpp
read_bin_file(std::string("/abs/path/to/ifm.bin"),
              reinterpret_cast<char*>(aie_ifm), ifm_size);

read_bin_file(std::string("/abs/path/to/wgt.bin"),
              reinterpret_cast<char*>(aie_wgt), wgt_size);

read_bin_file(std::string("/abs/path/to/cpu_ofm.bin"),
              reinterpret_cast<char*>(cpu_ofm), cpu_ofm_size);
```

If you **do not want to run correctness comparison**, you may:

* Skip loading `cpu_ofm.bin`
* Set `cpu_ofm_size = 0`

(Ensure the allocation logic remains consistent with your use case.)


#### 3.4 (Optional) Enable output comparison

This step is only required if you want to verify correctness against a golden output.

#### 3.5 Set tensor dimensions

```cpp
int Nin  = ..., Yin  = ..., Xin  = ..., Cin  = ...;
int Nout = ..., Yout = ..., Xout = ..., Cout = ...;
```

#### 3.6 Enable and configure comparison

Uncomment the compare block and set:

* `threshold` – absolute tolerance per element
* `read_aie_ofm_from_offset` – byte offset if OFM is written into a larger buffer object

Example:

```cpp
int read_aie_ofm_from_offset = 118784; // bytes
```


### 4. Place CERT Sim artifacts

Copy the generated artifacts to the expected locations:

#### 4.1 Control ELF

Place the subgraph-specific `control.elf` in:

```
Output/op_pdi_shape_input_0/Work_AIE4
```


#### 4.2 Parameter binary

Place `param.bin` in:

```
Output/op_pdi_shape_input_0
```

#### 4.3 Simulation support files

Copy the following from the `cert_sim` directory:

* `Makefile`
* `CERT_AIESIM_OPTIONS.txt`

into:

```
Output/op_pdi_shape_input_0
```



### 5. Run CERT Sim

```bash
cd Output/op_pdi_shape_input_0
make sim
```

This launches CERT Sim using CERT infrastructure for the configured subgraph.


## Notes & Common Pitfalls

* **Byte sizes must match exactly** — mismatches will silently corrupt data.
* Ensure `read_aie_ofm_from_offset` is correct if the subgraph writes OFM into a shared BO.
* Verify that the datatype used in `cmp_tensor_nyxc<T>` matches the actual tensor type.
* Always confirm that `LIST_OPS` matches the subgraph being tested.

---
