'''This program would manipulate the Work/ps/c_rts/systemC/Makefile to make it compatible for CertSim'''
import os, re

def get_cpp_basename(path: str) -> str:
    """Return the filename (without .cpp) of the single .cpp in a directory."""
    cpp = [f for f in os.listdir(path) if f.lower().endswith(".cpp")]
    if len(cpp) != 1: raise FileNotFoundError(f"Expected exactly one .cpp in {path!r}, found {len(cpp)}")
    return os.path.splitext(cpp[0])[0]

def _find_line_index(lines: list[str], pattern: str) -> int:
    """Return the first line index matching regex `pattern`, or -1 if not found."""
    rx = re.compile(pattern)
    for i, ln in enumerate(lines):
        if rx.search(ln):
            return i
    return -1

def _insert_after_first_token(line: str, token: str, text: str) -> str:
    """Insert `text` after the first standalone `token` in `line` (not inside paths/filenames)."""
    pat = rf'(^|\s)({re.escape(token)})(?=\s|\\|$)'
    m = re.search(pat, line)
    return line[:m.end(2)] + text + line[m.end(2):] if m else line


def patch_makefile(path_output: str) -> None:
    """Edit the SystemC Makefile per spec (opts, CERT path, link objs, and extra line)."""
    x = '''
cert_interface:
	(export LD_LIBRARY_PATH=${XILINX_VITIS_AIETOOLS}/lib/lnx64.o:${XILINX_VITIS_AIETOOLS}/tps/lnx64/gcc/lib64:$(LD_LIBRARY_PATH);"${XILINX_VITIS_AIETOOLS}/lnx64.o/tools/clang/bin/clang++" -fPIC -fpermissive -c -std=c++17 -D__AIE_ARCH__=42 -D__LOCK_FENCE_MODE__=2 -DAIE_OPTION_SCALAR_FLOAT_ON_VECTOR -Wno-enum-constexpr-conversion -Wno-error=reserved-user-defined-literal -Wno-format-security -Wno-deprecated-declarations -Werror=return-type -DSC_INCLUDE_DYNAMIC_PROCESSES -D__AIESIM__ -D__PS_INIT_AIE__ -DXAIE_DEBUG  -DUSE_CERT_LIBRARY=1  -D__AIECONTROLCODE__ -O0  -D main\(...\)=ps_main\(...\) -I${XILINX_VITIS_AIETOOLS}/include -I${XILINX_VITIS_AIETOOLS}/include/drivers/aiengine_aig -I${XILINX_HLS}/include -I${XILINX_VITIS_AIETOOLS}/tps/lnx64/gcc/include/c++/8.3.0 -I${XILINX_VITIS_AIETOOLS}/tps/lnx64/gcc/include/c++/8.3.0/backward -I${XILINX_VITIS_AIETOOLS}/tps/lnx64/gcc/include/c++/8.3.0/x86_64-pc-linux-gnu -I${XILINX_VITIS_AIETOOLS}/data/osci_systemc/include -I${XILINX_VITIS_AIETOOLS}/tps/boost_1_72_0 -I. -I$(AIE_SRC_DIR) -I${XILINX_VITIS_AIETOOLS}/include/xtlm/include -I${XILINX_VITIS_AIETOOLS}/include/common_cpp/common_cpp_v1_0/include -I${CERT_INTERFACE_PATH}/include -I "../../../.." -o "generated-objects/cert_interface.o" ${CERT_INTERFACE_PATH}/src/cert_interface.cpp -D__TEST_BENCH__=\"${TEST_BENCH}\" ) 

elf_loader:
	(export LD_LIBRARY_PATH=${XILINX_VITIS_AIETOOLS}/lib/lnx64.o:${XILINX_VITIS_AIETOOLS}/tps/lnx64/gcc/lib64:$(LD_LIBRARY_PATH);"${XILINX_VITIS_AIETOOLS}/lnx64.o/tools/clang/bin/clang++" -fPIC -fpermissive -c -std=c++17 -D__AIE_ARCH__=42 -D__LOCK_FENCE_MODE__=2 -DAIE_OPTION_SCALAR_FLOAT_ON_VECTOR -Wno-enum-constexpr-conversion -Wno-error=reserved-user-defined-literal -Wno-format-security -Wno-deprecated-declarations -Werror=return-type -DSC_INCLUDE_DYNAMIC_PROCESSES -D__AIESIM__ -D__PS_INIT_AIE__ -DXAIE_DEBUG  -DUSE_CERT_LIBRARY=1  -D__AIECONTROLCODE__ -O0  -D main\(...\)=ps_main\(...\) -I${XILINX_VITIS_AIETOOLS}/include -I${XILINX_VITIS_AIETOOLS}/include/drivers/aiengine_aig -I${XILINX_HLS}/include -I${XILINX_VITIS_AIETOOLS}/tps/lnx64/gcc/include/c++/8.3.0 -I${XILINX_VITIS_AIETOOLS}/tps/lnx64/gcc/include/c++/8.3.0/backward -I${XILINX_VITIS_AIETOOLS}/tps/lnx64/gcc/include/c++/8.3.0/x86_64-pc-linux-gnu -I${XILINX_VITIS_AIETOOLS}/data/osci_systemc/include -I${XILINX_VITIS_AIETOOLS}/tps/boost_1_72_0 -I. -I$(AIE_SRC_DIR) -I${XILINX_VITIS_AIETOOLS}/include/xtlm/include -I${XILINX_VITIS_AIETOOLS}/include/common_cpp/common_cpp_v1_0/include -I${CERT_INTERFACE_PATH}/include -I "../../../.." -o "generated-objects/elf_loader.o" ${CERT_INTERFACE_PATH}/src/elf_loader.cpp -D__TEST_BENCH__=\"${TEST_BENCH}\" ) 

'''
    makefile_path = os.path.join(path_output, "Work/ps/c_rts/systemC/Makefile")

    # load
    with open(makefile_path, "r", encoding="utf-8") as f:
        lines = f.readlines()

    # 1) add CERT_INTERFACE_PATH after TEST_BENCH= line and capture cpp basename
    i_tb = _find_line_index(lines, r"^\s*TEST_BENCH\s*=")
    if i_tb == -1:
        raise RuntimeError("TEST_BENCH= not found")
    # insert CERT_INTERFACE_PATH on the next line
    cert_line = 'CERT_INTERFACE_PATH=${XILINX_VITIS_AIETOOLS}/data/simmodels/osci/2.3.1/lnx64/8.3.0/systemc/protected/aiesim_cert_interface\n'
    insert_at = i_tb + 1
    lines.insert(insert_at, cert_line)
    # extract cpp path from TEST_BENCH= to compute cpp basename
    tb_val = os.path.join(path_output, "Work/ps/c_rts/systemC/generated-source")
    cpp_base = get_cpp_basename(tb_val)

    # 2) find rule "<cpp_base>:" and edit the next line (-O1→-O0 and add defines/includes)
    i_rule = _find_line_index(lines, rf"^\s*{re.escape(cpp_base)}\s*:")
    if i_rule == -1:
        raise RuntimeError(f"Rule '{cpp_base}:' not found")
    i_cmd = i_rule + 1
    if i_cmd >= len(lines):
        raise RuntimeError("Expected command line after rule not found")

    extras = " -DUSE_CERT_LIBRARY=1  -D__AIECONTROLCODE__ -I${CERT_INTERFACE_PATH}/include "
    cmd = lines[i_cmd]
    if "-O1" in cmd:
        cmd = cmd.replace("-O1", "-O0" + extras, 1)
    elif "-O0" in cmd:
        cmd = cmd.replace("-O0", "-O0" + extras, 1)
    else:
        # if no -O* present, just prepend -O0 and extras near the beginning after first space
        cmd = re.sub(r"(\s)", r"\1-O0" + extras + " ", cmd, count=1)
    lines[i_cmd] = cmd

    # 3) add blank line and then user-provided x after the edited line
    lines.insert(i_cmd + 2, ("" if x.endswith("\n") else "") + x + ("" if x.endswith("\n") else "\n"))

    # 4) after "link:" line, in the next line append "  cert_interface elf_loader  " after first <cpp_base>
    i_link = _find_line_index(lines, r"^\s*link\s*:")
    if i_link == -1:
        raise RuntimeError("link: not found")
    i_linkcmd = i_link
    if i_linkcmd >= len(lines):
        raise RuntimeError("Expected command line after link: not found")
    lcmd = lines[i_linkcmd]
    lcmd = _insert_after_first_token(lcmd, cpp_base, "  cert_interface elf_loader  ")
    lines[i_linkcmd] = lcmd


    # 5) add objs before "generated-objects/<cpp_base>.o"
    i_linkcmd = i_link + 1
    lcmd = lines[i_linkcmd]
    needle = f"generated-objects/{cpp_base}.o"
    inj = " generated-objects/cert_interface.o generated-objects/elf_loader.o -lelf "
    lcmd = lcmd.replace(needle, inj + needle, 1)
    lines[i_linkcmd] = lcmd

    # save
    with open(makefile_path, "w", encoding="utf-8") as f:
        f.writelines(lines)
        
    # Edit gen_wrapper.cpp to add CERT Sim initializing function
    cpp_path = os.path.join(tb_val, f"{cpp_base}.cpp")

    with open(cpp_path, "r", encoding="utf-8") as f:
        cpp_lines = f.readlines()
        
    include_statement = '''#endif
#if USE_CERT_LIBRARY == 1
#include "uc_sim_wrapper.h"
#else
#include "uc_runtime_handler.h"
'''

    cert_function = '''

#ifdef __AIESIM__
inline void executeCERTSim(std::string const& control_elf_path,
                           std::vector<uint32_t> const& columns,
                           std::map<std::string, patch> const& sym_tbl)
{
    const std::string cert_bin_path = "./Work/cert_medusa_info_console.bin";
    aiesim::array_init(cert_bin_path, 0X20000000000, 25, 20, 3, 4);
    hardware_context_id id = 0;
    id = aiesim::create_hardware_context(columns);
    aiesim::run_command(id, control_elf_path,
                        std::map<uint32_t, bool>{},
                        sym_tbl);
    aiesim::destroy_hardware_context(id);
}
#endif // __AIESIM__
'''

    # After '#include __TEST_BENCH__', insert y on the next line
    i_inc = _find_line_index(cpp_lines, r"#include\s+__TEST_BENCH__")
    if i_inc == -1: raise RuntimeError("'#include __TEST_BENCH__' not found in .cpp")
    cpp_lines.insert(i_inc + 1, include_statement)

    # Append z at end of file
    if cpp_lines and not cpp_lines[-1].endswith("\n"):
        cpp_lines[-1] += "\n"
    cpp_lines.append(cert_function)

    with open(cpp_path, "w", encoding="utf-8") as f:
        f.writelines(cpp_lines)
