#!/bin/bash

# AIE4 Automated Regression - Outer Wrapper
# Handles: job naming, output directory, email notifications, optional git branch switching

# Parse command-line arguments
LOCAL_BRANCH=""
REMOTE_BRANCH=""
EMAILS=""
SENDMAIL_PATH="/usr/sbin/sendmail"
OUTPUT_DIR=""
HWTEST=false
TARGET=""
BUILDTEST_OUTPUT_ROOT=""
EXTRA_ARGS=()

while [[ $# -gt 0 ]]; do
    case $1 in
        --local-branch)
            LOCAL_BRANCH="$2"
            shift 2
            ;;
        --remote-branch)
            REMOTE_BRANCH="$2"
            shift 2
            ;;
        --emails)
            EMAILS="$2"
            shift 2
            ;;
        --sendmail)
            SENDMAIL_PATH="$2"
            shift 2
            ;;
        --output-dir)
            OUTPUT_DIR="$2"
            shift 2
            ;;
        --target)
            TARGET="$2"
            shift 2
            ;;
        --buildtest-output-root)
            BUILDTEST_OUTPUT_ROOT="$2"
            shift 2
            ;;
        --hwtest)
            HWTEST=true
            shift
            ;;
        *)
            EXTRA_ARGS+=("$1")
            shift
            ;;
    esac
done

# Validate: can't use both
if [ -n "$LOCAL_BRANCH" ] && [ -n "$REMOTE_BRANCH" ]; then
    echo "Error: Cannot specify both --local-branch and --remote-branch"
    exit 1
fi

# script dir
SCRIPT_DIR=$(dirname "$(realpath "$0")")
REPO_DIR="$(realpath "$SCRIPT_DIR/..")"

# Create unique random job prefix
JOB_PREFIX="cron_$(date +%Y%m%d_%H%M%S)_$$_$RANDOM"

# Set default output directory if not provided
if [ -z "$OUTPUT_DIR" ]; then
    OUTPUT_DIR="/tmp/aie4_cron_$(date +%Y%m%d_%H%M%S)"
else
    # Validate that OUTPUT_DIR is an absolute path
    if [[ "$OUTPUT_DIR" != /* ]]; then
        echo "Error: --output-dir must be an absolute path (got: $OUTPUT_DIR)"
        echo "Example: --output-dir /tmp/my_test_output"
        echo "Or use: --output-dir \$PWD/my_test_output"
        exit 1
    fi
fi
mkdir -p "$OUTPUT_DIR"

# Create log file
LOG_FILE="$OUTPUT_DIR/cron_job.log"

# Flag to prevent duplicate cleanup
CLEANUP_DONE=0
ORIGINAL_BRANCH=""

# shellcheck disable=SC2329
cleanup() {
    # Prevent duplicate execution
    if [ $CLEANUP_DONE -eq 1 ]; then
        return
    fi
    CLEANUP_DONE=1

    echo "Cleaning up..."

    # Return to original branch if we switched
    if [ -n "$ORIGINAL_BRANCH" ]; then
        cd "$REPO_DIR" || true
        echo "Returning to original branch: $ORIGINAL_BRANCH"
        git checkout "$ORIGINAL_BRANCH" 2>/dev/null || true
    fi

    # Skip email if --emails was not provided
    if [ -z "$EMAILS" ]; then
        echo "Skipping email notification (no recipients specified)"
        return
    fi

    # Parse cron job log, and find the location of report
    REPORT_LOCN=$(grep "HTML report saved to:" "$OUTPUT_DIR/cron_job.log" | awk '{print $NF}')

    # Determine subject based on report status (not bash exit code which may be misleading)
    if [ -f "$REPORT_LOCN" ] && grep -q "✅ SUCCESS" "$REPORT_LOCN"; then
        SUBJECT="AIE4 Cron Job SUCCESS - $(date +%Y%m%d_%H%M%S)"
    else
        SUBJECT="AIE4 Cron Job FAILED - $(date +%Y%m%d_%H%M%S)"
    fi

    # Send HTML email with proper MIME headers using sendmail
    # Example: --emails "user1@amd.com,user2@amd.com,user3@amd.com"
    echo "Sending email to: $EMAILS using $SENDMAIL_PATH"
    (
        echo "To: $EMAILS"
        echo "Subject: $SUBJECT"
        echo "Content-Type: text/html; charset=UTF-8"
        echo "MIME-Version: 1.0"
        echo ""
        if [ -f "$REPORT_LOCN" ]; then
            cat "$REPORT_LOCN"
        fi
    ) | "$SENDMAIL_PATH" -t
}

# Set trap for cleanup on exit
trap cleanup EXIT INT TERM

echo "========================================"
echo "AIE4 Automated Regression Starting"
echo "Timestamp: $(date)"
echo "Output Directory: $OUTPUT_DIR"
echo "Job Prefix: $JOB_PREFIX"
if [ -n "$EMAILS" ]; then
    echo "Email Recipients: $EMAILS"
    echo "Sendmail Path: $SENDMAIL_PATH"
else
    echo "Email: Disabled (no recipients specified)"
fi
if [ -n "$LOCAL_BRANCH" ]; then
    echo "Git: Switching to local branch '$LOCAL_BRANCH'"
elif [ -n "$REMOTE_BRANCH" ]; then
    echo "Git: Switching to remote branch '$REMOTE_BRANCH'"
else
    echo "Git: Using current branch (no switch)"
fi
echo "========================================"

# Optional: Switch to branch
if [ -n "$LOCAL_BRANCH" ] || [ -n "$REMOTE_BRANCH" ]; then
    cd "$REPO_DIR" || exit 1

    # Check for uncommitted changes
    if [ -n "$(git status --porcelain --untracked-files=no)" ]; then
        echo "Error: You have uncommitted changes. Please commit or stash them first."
        git status --short --untracked-files=no
        exit 1
    fi

    # Save current branch
    ORIGINAL_BRANCH=$(git rev-parse --abbrev-ref HEAD 2>/dev/null || echo "")
    echo "Current branch: ${ORIGINAL_BRANCH:-detached HEAD}"

    if [ -n "$LOCAL_BRANCH" ]; then
        # Switch to local branch
        echo "Switching to local branch: $LOCAL_BRANCH"
        if git rev-parse --verify "$LOCAL_BRANCH" >/dev/null 2>&1; then
            git checkout "$LOCAL_BRANCH" || { echo "Failed to checkout branch"; exit 1; }
        else
            echo "Error: Local branch '$LOCAL_BRANCH' does not exist"
            exit 1
        fi
    else
        # Switch to remote branch (format: remote/branch)
        if [[ "$REMOTE_BRANCH" =~ ^([^/]+)/(.+)$ ]]; then
            REMOTE_NAME="${BASH_REMATCH[1]}"
            BRANCH_NAME="${BASH_REMATCH[2]}"

            echo "Fetching from $REMOTE_NAME..."
            git fetch "$REMOTE_NAME" || { echo "Failed to fetch from $REMOTE_NAME"; exit 1; }

            if git rev-parse --verify "$REMOTE_NAME/$BRANCH_NAME" >/dev/null 2>&1; then
                echo "Checking out $REMOTE_NAME/$BRANCH_NAME..."
                git checkout -B "$BRANCH_NAME" "$REMOTE_NAME/$BRANCH_NAME" || { echo "Failed to checkout branch"; exit 1; }
                git submodule update --init --recursive
            else
                echo "Error: Remote branch '$REMOTE_NAME/$BRANCH_NAME' does not exist"
                exit 1
            fi
        else
            echo "Error: Remote branch must be in format 'remote/branch' (e.g., 'origin/main')"
            exit 1
        fi
    fi
fi

# check if output directory exists
if [ ! -d "$OUTPUT_DIR" ]; then
  echo "Error: Output directory does not exist!"
  exit 1
fi

# check if the script exists
if [ ! -f "$SCRIPT_DIR/run_lsf_tests.sh" ]; then
  echo "Error: run_lsf_tests.sh not found!"
  exit 1
fi

# clean the output folder
rm -rf "$REPO_DIR"/Output/*

# Build command with optional flags
HWTEST_FLAG=""
if [ "$HWTEST" = true ]; then
    HWTEST_FLAG="--hwtest"
fi

TARGET_FLAG=""
if [ -n "$TARGET" ]; then
    TARGET_FLAG="--target $TARGET"
fi

BUILDTEST_OUTPUT_ROOT_FLAG=""
if [ -n "$BUILDTEST_OUTPUT_ROOT" ]; then
    # Validate that BUILDTEST_OUTPUT_ROOT is an absolute path
    if [[ "$BUILDTEST_OUTPUT_ROOT" != /* ]]; then
        echo "Error: --buildtest-output-root must be an absolute path (got: $BUILDTEST_OUTPUT_ROOT)"
        echo "Example: --buildtest-output-root /tmp/my_test_output"
        echo "Or use: --buildtest-output-root \$PWD/my_test_output"
        exit 1
    fi
    BUILDTEST_OUTPUT_ROOT_FLAG="--output-root $BUILDTEST_OUTPUT_ROOT"
fi

# Run the test orchestrator script
echo "Running: bash $SCRIPT_DIR/run_lsf_tests.sh --job-prefix $JOB_PREFIX --output-dir $OUTPUT_DIR $HWTEST_FLAG $TARGET_FLAG $BUILDTEST_OUTPUT_ROOT_FLAG ${EXTRA_ARGS[*]}"
echo "Outputs will be written to $LOG_FILE"
bash "$SCRIPT_DIR/run_lsf_tests.sh" --job-prefix "$JOB_PREFIX" --output-dir "$OUTPUT_DIR" $HWTEST_FLAG $TARGET_FLAG $BUILDTEST_OUTPUT_ROOT_FLAG "${EXTRA_ARGS[@]}" > "$LOG_FILE" 2>&1
exit_code=$?

echo "========================================"
echo "AIE4 Automated Regression Finished"
echo "Exit Code: $exit_code"
echo "Timestamp: $(date)"
echo "========================================"

exit $exit_code