""""""
import argparse
from pathlib import Path
import hashlib
import os
import subprocess


def md5sum(file_path: Path, chunk_size=8192):
    """
    Calculate the MD5 checksum of the specified file.
    """
    md5 = hashlib.md5()
    with file_path.open("rb") as f:
        for chunk in iter(lambda: f.read(chunk_size), b""):
            md5.update(chunk)
    return md5.hexdigest()


def generate_html(patch_file: Path, html_file: Path, title: str = "DMA HPP") -> bool:
    """
    Generate an HTML file from a patch file.
    """
    try:
        subprocess.run(
            [
                "npx", "--yes", "diff2html-cli",
                "-i", "file",
                "-s", "side",          # side-by-side layout
                "-F", str(html_file),  # output HTML file
                "-t", title,           # title
                "--", str(patch_file)
            ],
            check=False
        )
        return True
    except Exception:  # pylint: disable=broad-except
        return False


def generate_index_html(diff_results: list, output_file: Path):
    """Generate index.html with links to all diffs"""
    html_content = """<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>dma.hpp diffs</title>
</head>
<body>
    <h1>dma.hpp diffs</h1>
    <ul>
"""

    for result in diff_results:
        operator = result['operator']
        html_file = result['html_file']
        html_content += f'<li><a href="{html_file.name}">{operator}</a></li>\n'

    html_content += """    </ul>
</body>
</html>
"""

    with output_file.open('w', encoding='utf-8') as f:
        f.write(html_content)


def main(dir1: str, dir2: str, patch_dir: Path):
    """Main function"""
    succeeded = True
    diff_results = []

    patch_dir.mkdir(parents=True, exist_ok=True)
    patch_file = Path(f"{patch_dir}/all.patch")
    html_file = Path(f"{patch_dir}/all_diffs.html")

    if patch_file.exists():
        patch_file.unlink()

    with patch_file.open("ab") as patch:
        for odir in os.listdir(dir1):
            dma_1 = Path(f'{dir1}/{odir}/dma.hpp')
            if not dma_1.exists():
                print(f"File doesn't exist on origin/main: {str(dma_1)}")
                succeeded = False
                continue
            if dma_1.exists() and dma_1.stat().st_size == 0:
                print(f"File exists in origin/main but is empty: {str(dma_1)}")
                succeeded = False
                continue
            md5sum_1 = md5sum(dma_1)

            dma_2 = Path(f'{dir2}/{odir}/dma.hpp')
            if not dma_2.exists():
                md5sum_2 = "NOT_FOUND"
            else:
                md5sum_2 = md5sum(dma_2)

            if md5sum_1 != md5sum_2:
                print(f'Files are different: {str(dma_1)} and {str(dma_2)}')
                second_file = str(dma_2) if dma_2.exists() else "/dev/null"
                subprocess.run(
                    ["git", "diff", "--no-index", str(dma_1), second_file],
                    stdout=patch,
                    stderr=subprocess.DEVNULL,
                    check=False
                )

                # Create individual patch file
                curr_patch_file = patch_dir / f"{odir}.patch"
                curr_html_file = patch_dir / f"{odir}.html"
                with curr_patch_file.open("wb") as current_patch:
                    subprocess.run(
                        ["git", "diff", "--no-index", str(dma_1), second_file],
                        stdout=current_patch,
                        stderr=subprocess.DEVNULL,
                        check=False
                    )

                    if generate_html(curr_patch_file, curr_html_file, f"DMA HPP - {odir}"):
                        diff_results.append({
                            'operator': odir,
                            'html_file': curr_html_file})

                succeeded = False
                continue
            # print(f'Files are identical: {str(dma_1)} and {str(dma_2)}')

        for odir in os.listdir(dir2):
            dma_2 = Path(f'{dir2}/{odir}/dma.hpp')
            if not dma_2.exists():
                print(f"File doesn't exist in PR branch: {str(dma_2)}")
                succeeded = False
                continue
            if dma_2.exists() and dma_2.stat().st_size == 0:
                print(f"File exists in PR branch but is empty: {str(dma_2)}")
                succeeded = False
                continue
            if dma_2.exists() and not Path(f'{dir1}/{odir}/dma.hpp').exists():
                print(f"Detected new file in PR branch: {str(dma_2)}")

    # Generate index.html
    generate_index_html(diff_results, html_file)
    print(f"\nIndex generated: {str(html_file)}")
    print("Comparison succeeded" if succeeded else "Comparison failed")


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Compare dma.hpp from two directories')
    parser.add_argument('dir1', type=str, help='First directory')
    parser.add_argument('dir2', type=str, help='Second directory')
    parser.add_argument('patch_dir', type=Path, help='Patch directory')
    args = parser.parse_args()
    main(args.dir1, args.dir2, args.patch_dir)
