#!/bin/bash
set -e

# AIE4 Test Runner - Environment Setup and Execution
# This script sets up the environment and runs the Python test orchestrator
# Can be used standalone for manual testing or via ci_cron_job.sh for automation

# shellcheck disable=SC2329
cleanup() {
    echo "Cleaning up..."
}

# Set trap for cleanup on exit
trap cleanup EXIT INT TERM

# Configuration
REPO_DIR="$(realpath "$(dirname "${BASH_SOURCE[0]}")/..")"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PYTHON_SCRIPT="${PYTHON_SCRIPT:-$SCRIPT_DIR/run_lsf_tests.py}"

# source lsf (disable set -e temporarily as LSF profile may return non-zero)
set +e
source /group/xsjfarm/lsf/conf/profile.lsf
set -e

# check if we can submit a job, permissions
if ! bjobs -u "$USER" >/dev/null 2>&1; then
    echo "Error: You do not have permission to use LSF (bjobs command failed)"
    exit 1
fi

echo "========================================"
echo "AIE4 Test Runner Starting"
echo "Timestamp: $(date)"
echo "Repository: $REPO_DIR"
echo "========================================"

# Change to repository directory
cd "$REPO_DIR" || exit 1

# Source environment settings
if [ -f "$REPO_DIR/settings.sh" ]; then
    echo "Sourcing settings.sh..."
    set +e
    source "$REPO_DIR/settings.sh"
    set -e
else
    echo "Warning: settings.sh not found"
fi

# If virtual environment does not exist, create it
if [ ! -d "$REPO_DIR/env" ] && [ -z "$VENV_PATH" ]; then
    echo "Creating virtual environment..."
    /tool/pandora64/bin/python3.10 -m venv env
fi

# Activate virtual environment (prefer VENV_PATH if set - it's the venv directory)
if [ -n "$VENV_PATH" ] && [ -f "$VENV_PATH/bin/activate" ]; then
    echo "Activating virtual environment from VENV_PATH: $VENV_PATH"
    source "$VENV_PATH/bin/activate"
elif [ -f "$REPO_DIR/env/bin/activate" ]; then
    echo "Activating virtual environment from $REPO_DIR/env/bin/activate"
    source "$REPO_DIR/env/bin/activate"
else
    echo "Error: Virtual environment not found at $REPO_DIR/env/bin/activate"
    exit 1
fi

# Install/update prerequisites (skip if VENV_PATH was provided - assume deps installed)
echo "Installing prerequisites..."
if [ -z "$VENV_PATH" ]; then
    pip install -q -r requirements.txt
fi

# Set AIE4_ROOT_DIR for Python script
export AIE4_ROOT_DIR="$REPO_DIR"

# Run the Python script
echo "========================================"
echo "Starting Python test orchestrator..."
echo "Command: python $PYTHON_SCRIPT $*"
echo "========================================"
python "$PYTHON_SCRIPT" "$@"

# Capture exit code
EXIT_CODE=$?

echo "========================================"
echo "AIE4 Test Runner Finished"
echo "Exit Code: $EXIT_CODE"
echo "Timestamp: $(date)"
echo "========================================"

exit $EXIT_CODE
