#!/bin/bash

set -e  # Exit on error

# Cleanup function
cleanup() {
    local exit_code=$?

    # Return to original branch if we saved it
    if [ -n "${ORIGINAL_BRANCH:-}" ]; then
        git checkout "$ORIGINAL_BRANCH" 2>/dev/null || true
    fi

    # Restore original Output directory if it existed
    if [ "${renamed:-0}" -eq 1 ] && [ -n "${temp_dir:-}" ] && [ -d "$temp_dir" ]; then
        echo "Restoring original Output directory..."
        rm -rf Output
        mv "$temp_dir" Output
        echo "Restored $temp_dir to Output"
    fi

    #rm -rf ci-artifacts
    exit $exit_code
}

# Set trap for cleanup on exit
trap cleanup EXIT INT TERM

# Check if argument is provided
if [ $# -eq 0 ]; then
    cat <<EOF
Error: IPSP remote name on your local workspace is required e.g., :

(env) bash-4.4$ git remote -v
origin  git@gitenterprise.xilinx.com:IPSP/aie4_models.git (fetch)

if IPSP remote name is 'origin' then run the script as follows
bash $0 origin
EOF
    exit 1
fi

REMOTE_NAME="$1"

# Pre-flight checks
echo "Fetching from $REMOTE_NAME..."
git fetch "$REMOTE_NAME" || { echo "Failed to fetch from $REMOTE_NAME"; exit 1; }

# Check if remote/main exists
if git rev-parse --verify "$REMOTE_NAME/main" >/dev/null 2>&1; then
    echo "✓ $REMOTE_NAME/main exists"
else
    echo "✗ Error: $REMOTE_NAME/main does not exist"
    exit 1
fi

# Check if there are any uncommitted changes
if [ -n "$(git status --porcelain --untracked-files=no)" ]; then
    echo "Error: You have uncommitted changes"
    git status --short --untracked-files=no
    exit 1
fi

# Save current branch (if on one)
ORIGINAL_BRANCH=$(git rev-parse --abbrev-ref HEAD)
echo "Current branch: $ORIGINAL_BRANCH"

# Backup existing Output directory if it exists
temp_dir="Output_backup_$(date +%Y%m%d_%H%M%S)"
renamed=0
if [ -d "Output" ]; then
    mv Output "$temp_dir"
    echo "Backed up existing Output to $temp_dir"
    renamed=1
fi

# remove artifacts
rm -rf ci-artifacts

# create artifacts directory
mkdir -p ci-artifacts/logs

# run dma tests
echo "=== Running DMA tests on current branch ($ORIGINAL_BRANCH) ==="
cd buildtest
pytest -m dma -vv --tb=auto 2>&1 | tee ../ci-artifacts/logs/pytest_New.log
cd ..
if [ ! -d "Output" ]; then
    echo "✗ Error: Output directory not created by pytest"
    exit 1
fi
mv Output ci-artifacts/Output_New
echo "✓ Saved output from current branch"

# run dma tests on main
echo "=== Switching to $REMOTE_NAME/main ==="
git checkout ${REMOTE_NAME}/main
echo "=== Running DMA tests on $REMOTE_NAME/main ==="
rm -rf Output
cd buildtest
pytest -m dma -vv --tb=auto 2>&1 | tee ../ci-artifacts/logs/pytest_Old.log
cd ..
if [ ! -d "Output" ]; then
    echo "✗ Error: Output directory not created by pytest"
    exit 1
fi
mv Output ci-artifacts/Output_Old
echo "✓ Saved output from $REMOTE_NAME/main"

# Return to previous branch/commit
echo "=== Returning to $ORIGINAL_BRANCH ==="
git checkout -

# Compare
echo "=== Comparing outputs ==="
python ci/compare_output.py ci-artifacts/Output_Old ci-artifacts/Output_New ci-artifacts > ci-artifacts/logs/compare_output.log

echo "=== Comparison Results ==="
cat ci-artifacts/logs/compare_output.log
