# Getting Started

## Installation

Please refer to the main [README](https://gitenterprise.xilinx.com/IPSP/aie4_models/blob/main/README.md) for complete installation instructions including:

- Prerequisites and system requirements
- Conda environment setup
- Building and running operators
- Environment configuration

## Quick Start

Once you have completed the setup from the README, you can start using the Graph module for memory allocation.

## Basic Usage

The Graph module provides memory allocation and scheduling for tensor operations. Here's a minimal example:

```python
from graph.allocation_types import AllocationConfig, AllocationStrategy, XrtId, MemoryBlock
from graph.multibin_allocator import MultiBinGraphMemoryScheduler

# Configure memory allocation strategy
config = AllocationConfig(
    strategy=AllocationStrategy.FIRST_FIT,
    bin=XrtId.DEFAULT
)

# Define available memory blocks
blocks = [MemoryBlock(start=0, size=1024, is_free=True)]

# Build scheduler
scheduler = MultiBinGraphMemoryScheduler.build([(config, blocks)])
```

!!! note "Complete Examples"
    For comprehensive, up-to-date examples, see the test files:

    - [`test_multibin_allocator.py`](https://gitenterprise.xilinx.com/IPSP/aie4_models/blob/main/graph/tests/test_multibin_allocator.py) - Multi-bin scheduling examples
    - [`test_tensor_memory_allocator.py`](https://gitenterprise.xilinx.com/IPSP/aie4_models/blob/main/graph/tests/test_tensor_memory_allocator.py) - Tensor allocation with spilling
    - [`test_continuous_memory_allocator.py`](https://gitenterprise.xilinx.com/IPSP/aie4_models/blob/main/graph/tests/test_continuous_memory_allocator.py) - Base allocation strategies

    These test files are always kept in sync with the latest API changes.

## Running Tests

```bash
# Run all tests (includes unit tests and doctests)
pytest

# Run graph tests only
pytest graph/tests/

# Run doctests only
pytest --doctest-modules graph/

# Run with coverage
bash coverage.sh

# Run specific test file
pytest graph/tests/test_multibin_allocator.py -v
```

## Documentation Examples

All API documentation includes **doctests** - executable examples that are automatically tested. These ensure the documentation always reflects working code:

```python
# Example from allocation_types.py docstring:
>>> from graph.allocation_types import MemoryBlock
>>> block = MemoryBlock(start=0, size=1024, is_free=True)
>>> block.start
0
>>> block.size
1024
```

!!! tip "Doctests as Examples"
    Every class and function in the Graph API has doctest examples in its docstring. These are:

    - **Automatically tested** by pytest (via `--doctest-modules`)
    - **Always up-to-date** (tests fail if API changes)
    - **Copy-pasteable** code you can use directly

    View the source code or hover over functions in your IDE to see these examples!

## Next Steps

- Explore the [Graph API](../api/graph/index.md) for detailed documentation
- Check out the test files in `graph/tests/` for more examples
- Read about [allocation strategies](../api/graph/allocation-types.md)
- See [CONTRIBUTING.md](https://gitenterprise.xilinx.com/IPSP/aie4_models/blob/main/CONTRIBUTING.md) for development guidelines
