"""Create a command line argument parser from a Pydantic model."""

from enum import Enum
from typing import Type, get_origin, Literal, get_args
import argparse
from pydantic import BaseModel


class TypedBaseModel(BaseModel):
    """Base class for Pydantic models that can be used to generate an argparse parser."""
    model_config = {"populate_by_name": True}


def build_typed_parser(cls: Type[TypedBaseModel], description: str = "") -> argparse.ArgumentParser:
    """Build an argparse parser from a Pydantic model.

    Args:
        cls: Pydantic BaseModel class to generate parser from
        description: Optional description for the argument parser

    Returns:
        Configured ArgumentParser instance
    """
    parser = argparse.ArgumentParser(description=description or cls.__doc__)

    for field_name, field_info in cls.model_fields.items():
        # Build argument names
        arg_names = [f"--{field_name}"]

        # Add short alias if present
        if field_info.alias:
            arg_names.append(f"-{field_info.alias}")

        kwargs = {
            "help": field_info.description,
            "dest": field_name  # Ensure argparse uses field_name
        }

        # Handle Enum types
        if isinstance(field_info.annotation, type) and issubclass(field_info.annotation, Enum):
            enum_class = field_info.annotation
            kwargs["choices"] = [e.value for e in enum_class]
            kwargs["default"] = field_info.default.value
            kwargs["type"] = str
        # Handle bool
        elif field_info.annotation is bool:
            value = field_info.default
            if value is None or value is False:
                kwargs["action"] = "store_true"
                kwargs["default"] = False
            else:
                kwargs["action"] = "store_false"
                kwargs["default"] = True
        # Handle Literal
        elif get_origin(field_info.annotation) is Literal:
            kwargs["choices"] = get_args(field_info.annotation)
            kwargs["default"] = field_info.default
            kwargs["type"] = type(field_info.default)
        # Regular types
        else:
            kwargs["type"] = field_info.annotation
            kwargs["default"] = field_info.default

        parser.add_argument(*arg_names, **kwargs)

    return parser
