#!/bin/bash
set -e  # Exit on error
set -o pipefail  # Catch errors in pipes

# Get the directory where this script is located
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

# create unique output directory
OUTPUT_DIR="$SCRIPT_DIR/Output_$(date +%Y%m%d_%H%M%S)"

# list of pytests that we wish to run (can be overridden by environment variable)
PYTESTS="${PYTESTS:-(test_conv or test_gemm) and ([0] or -0])}"

# Number of pytest xdist workers (can be overridden by environment variable)
PYTEST_XDIST_WORKER_COUNT="${PYTEST_XDIST_WORKER_COUNT:-2}"

# hardware host IP (set default or override with environment variable)
HW_HOST="${HW_HOST:-10.228.200.87}"

# clean directory
cleandir() {
    dir="$*"
    if [ -d "$dir" ]; then
        # Kill all processes using the directory
        lsof +D "$dir" 2>/dev/null | awk 'NR>1 {print $2}' | sort -u | xargs -r kill -9 2>/dev/null || true
        sleep 2
        # Force unmount if needed (for NFS)
        sync
        # Remove directory with force
        rm -rf "$dir" 2>/dev/null || true
        # If still exists, try moving it aside
        if [ -d "$dir" ]; then
            mv "$dir" "${dir}.old.$$" 2>/dev/null || true
            rm -rf "${dir}.old.$$" 2>/dev/null &
        fi
    fi
}

# Cleanup function
cleanup() {
    local exit_code=$?
    cleandir "$OUTPUT_DIR"
    exit $exit_code
}

# Set trap for cleanup on exit
trap cleanup EXIT INT TERM

# Change to script directory
cd "$SCRIPT_DIR"
echo "Running pytests in directory: $(pwd)"

# Activate virtual environment if it exists
if [ -f "settings.sh" ]; then
    set +e
    source settings.sh
    set -e
fi

if [ -d "env" ] && [ -f "env/bin/activate" ]; then
    source env/bin/activate
fi

# change to buildtest
cd "$SCRIPT_DIR/buildtest"

# generate all bins in parallel
# --capture=no (-s): Don't capture stdout/stderr to reduce memory usage in pytest controller
python -m pytest -m dma -k "$PYTESTS" --target cert --output-root "$OUTPUT_DIR/Output_{{uuid}}" --dist=loadfile -n "${PYTEST_XDIST_WORKER_COUNT}" --capture=no

# Flatten the directory structure: move Output_gw*/op_* to OUTPUT_DIR/op_*
echo "Flattening directory structure..."
for worker_dir in "$OUTPUT_DIR"/Output_*/; do
    if [ -d "$worker_dir" ]; then
        # Move all contents from worker directory up one level
        mv "$worker_dir"* "$OUTPUT_DIR/" 2>/dev/null || true
        # Remove now-empty worker directory
        rm -rf "$worker_dir" 2>/dev/null || true
    fi
done

# Group operators by DTYPE_ACT and run hardware tests per group
echo "Grouping operators by DTYPE_ACT and running hardware tests..."
cd "$SCRIPT_DIR"
python HW_requirements/group_by_dtype_and_validate.py --output-dir "$OUTPUT_DIR" --hw-host "$HW_HOST"
