import os
import sys
CURRDIR = os.path.dirname(os.path.abspath(__file__))
WORKDIR = os.path.join(CURRDIR, 'Work')
HOSTDIR = os.path.join(CURRDIR, '..' , '..', 'host')

import subprocess
from typing import List

from kerneltest.maxpool.maxpool_dataflow import generate_maxpool_dataflow
from kerneltest.overlay_1x1 import overlay_stack_size, overlay_heap_size

def aiecompiler_args(
    Yi: int, Xi: int,
    Yo: int, Xo: int, Co: int,
    Yis: int, Xis: int,
    Yos: int, Xos: int, Cs: int,
    Ky: int, Kx: int,
    Py: int, Px: int,
    Sy: int, Sx: int,
    sign: int,
    txn_mode: int,
) -> List[str]:
    host_filename = HOSTDIR + '/maxpool.cpp'
    return [
        'aiecompiler',
        host_filename,
        '-v',
        '--disable-multirate-analysis',
        '--part=xc10MDS1',
        '--adf-api-log-level=5',
        '-log-level=5',
        '--disable-dma-autostart=true',
        '--enable-core-processor-bus=true',
        f'--workdir={WORKDIR}',
        f'--stacksize={overlay_stack_size()}',
        f'--heapsize={overlay_heap_size()}',
        f'--include={CURRDIR}',
        f'--include={HOSTDIR}',
        f'--include={os.path.join(CURRDIR, "..", "..", "kernel")}',
        f'--include={os.path.join(CURRDIR, "..", "..", "kernel/common")}',
        f'--Xpreproc="-DY_IN={Yi}"',
        f'--Xpreproc="-DX_IN={Xi}"',
        f'--Xpreproc="-DC_OUT={Co}"',
        f'--Xpreproc="-DY_OUT={Yo}"',
        f'--Xpreproc="-DX_OUT={Xo}"',
        f'--Xpreproc="-DYIS={Yis}"',
        f'--Xpreproc="-DXIS={Xis}"',
        f'--Xpreproc="-DCS={Cs}"',
        f'--Xpreproc="-DYOS={Yos}"',
        f'--Xpreproc="-DXOS={Xos}"',
        f'--Xpreproc="-DKERNEL_Y={Ky}"',
        f'--Xpreproc="-DKERNEL_X={Kx}"',
        f'--Xpreproc="-DSTRIDE_Y={Sy}"',
        f'--Xpreproc="-DSTRIDE_X={Sx}"',
        f'--Xpreproc="-DPAD_Y={Py}"',
        f'--Xpreproc="-DPAD_X={Px}"',
        f'--Xpreproc="-DSIGN={sign}"',
        f'--Xpreproc="-DTXN_MODE={txn_mode}"',
        f'--Xpreproc="-DMAXPOOL_NOQDQ_A8={1}"',
    ]

def aiesimulator_args() -> List[str]:
    return [
        'aiesimulator',
        '--dump-vcd=trace',
        '--profile',
        f'--pkg-dir={WORKDIR}',
    ]

def main():
    assert len(sys.argv) in (1, 3)
    txn_mode = int(sys.argv[1])
    shape_index = int(sys.argv[2]) if len(sys.argv) == 3 else None
    shape_table = [
        (
            (2, 2, 64),    # Yo, Xo, Co
            (2, 2, 64),    # Yos, Xos, Cs
            (3, 3),         # Yi, Xi
            (3, 3),         # Yis, Xis
            (2, 2),         # Ky, Kx
            (0, 0),         # Py, Px
            (1, 1),         # Sy, Sx
            0,              # Sign
        ),
        (
            (1, 1, 64),    # Yo, Xo, Co
            (1, 1, 64),    # Yos, Xos, Cs
            (2, 2),         # Yi, Xi
            (2, 2),         # Yis, Xis
            (2, 2),         # Ky, Kx
            (0, 0),         # Py, Px
            (1, 1),         # Sy, Sx
            1,              # Sign
        ),
        (
            (2, 2, 128),    # Yo, Xo, Co
            (2, 2, 128),    # Yos, Xos, Cs
            (4, 4),         # Yi, Xi
            (4, 4),         # Yis, Xis
            (3, 3),         # Ky, Kx
            (0, 0),         # Py, Px
            (1, 1),         # Sy, Sx
            1,              # Sign
        ),
        (
            (9, 9, 128),    # Yo, Xo, Co
            (9, 9, 128),    # Yos, Xos, Cs
            (19, 19),         # Yi, Xi
            (19, 19),         # Yis, Xis
            (3, 3),         # Ky, Kx
            (0, 0),         # Py, Px
            (2, 2),         # Sy, Sx
            1,              # Sign
        ),
        (
            (7, 19, 64),    # Yo, Xo, Co
            (7, 19, 64),    # Yos, Xos, Cs
            (15, 39),         # Yi, Xi
            (15, 39),         # Yis, Xis
            (3, 3),         # Ky, Kx
            (0, 0),         # Py, Px
            (2, 2),         # Sy, Sx
            0,              # Sign
        ),
    ]
    if shape_index is not None:
        shape_table = [shape_table[shape_index]]
    for (Yo, Xo, Co), (Yos, Xos, Cs), (Yi, Xi),  (Yis, Xis), (Ky, Kx), (Py, Px), (Sy, Sx), sign in shape_table:
        generate_maxpool_dataflow(Yi, Xi, Yo, Xo, Co, Yis, Xis, Yos, Xos, Cs, Ky, Kx, Py, Px, Sy, Sx, txn_mode)
        compile_args = aiecompiler_args(Yi, Xi, Yo, Xo, Co, Yis, Xis, Yos, Xos, Cs, Ky, Kx, Py, Px, Sy, Sx, sign, txn_mode)
    if os.path.isdir(WORKDIR):
        print("Deleting existing Work folder")
        subprocess.run(
            "rm -r Work",
            shell=True
        )
    os.makedirs(WORKDIR)
    os.chdir(WORKDIR)
    subprocess.run(compile_args)
    subprocess.run(
        f"sed -i 's/-ladf_api/-ladf_rt_ctrl_api -ladf_api/g' {WORKDIR}/ps/c_rts/systemC/Makefile",
        shell=True
    )
    subprocess.run(
        f'make -C {WORKDIR}/ps/c_rts/systemC/ all',
        shell=True
    )
    aiesim_args = aiesimulator_args()
    subprocess.run(aiesim_args)

if __name__ == '__main__':
    main()
