import os
import sys

CURRDIR = os.path.dirname(os.path.abspath(__file__))
sys.path.append(os.path.join(CURRDIR, '..', '..', 'dmacompiler'))
HOSTDIR = os.path.join(CURRDIR, '..' , '..', 'host')
WORKDIR = os.path.join(CURRDIR, 'Work')

import subprocess
from typing import List

from kerneltest.overlay_1x1 import (overlay_stack_size, overlay_heap_size)
from dmacompiler import BackEnd

from resize_nni_helpers import (
    CoordinateTransfromationMode,
    ResizeNNIDims,
    resize_nni_input_subvol_dims,
)

from resize_nni_dataflow import (
    compile_resize_nni_dataflow,
)

def aiecompiler_args(
    dims: ResizeNNIDims,
    run_mode: str,
) -> List[str]:
    host_filename = HOSTDIR + '/resize_nni.cpp'
    half_pixel = 1 if (dims.mode == CoordinateTransfromationMode.HALF_PIXEL.value) else 0
    align_corners = 1 if (dims.mode == CoordinateTransfromationMode.ALIGN_CORNERS.value) else 0
    asymmetric = 1 if (dims.mode == CoordinateTransfromationMode.ASYMMETRIC.value) else 0
    return [
        'aiecompiler',
        host_filename,
        '-v',
        '--disable-multirate-analysis',
        '--part=xc10MDS1',
        '--adf-api-log-level=5',
        '-log-level=5',
        '--disable-dma-autostart=true',
        '--enable-core-processor-bus=true',
        f'--workdir={WORKDIR}',
        f'--stacksize={overlay_stack_size()}',
        f'--heapsize={overlay_heap_size()}',
        f'--include={CURRDIR}',
        f'--include={HOSTDIR}',
        f'--include={os.path.join(CURRDIR, "..", "..", "kernel")}',
        f'--include={os.path.join(CURRDIR, "..", "..", "kernel/common")}',
        f'--include={os.path.join(CURRDIR, "..", "..", "kernel/resize_nni")}',
        f'--Xpreproc="-DHALF_PIXEL={half_pixel}"',
        f'--Xpreproc="-DALIGN_CORNERS={align_corners}"',
        f'--Xpreproc="-DASYMMETRIC={asymmetric}"',
        f'--Xpreproc="-DCIN={dims.Ci}"',
        f'--Xpreproc="-DYIN={dims.Yi}"',
        f'--Xpreproc="-DXIN={dims.Xi}"',
        f'--Xpreproc="-DCOUT={dims.Co}"',
        f'--Xpreproc="-DYOUT={dims.Yo}"',
        f'--Xpreproc="-DXOUT={dims.Xo}"',
        f'--Xpreproc="-DCIS={dims.Cis}"',
        f'--Xpreproc="-DYIS={dims.Yis}"',
        f'--Xpreproc="-DXIS={dims.Xis}"',
        f'--Xpreproc="-DCOS={dims.Cos}"',
        f'--Xpreproc="-DYOS={dims.Yos}"',
        f'--Xpreproc="-DXOS={dims.Xos}"',
        f'--Xpreproc="-DYIS_STEP={dims.Yis_step}"',
        f'--Xpreproc="-DXIS_STEP={dims.Xis_step}"',
        f'--Xpreproc="-DYIS_OFFSET={dims.Yis_offset}"',
        f'--Xpreproc="-DXIS_OFFSET={dims.Xis_offset}"',
        f'--Xpreproc="-DASM_MODE={1 if run_mode == "cert" else 0}"',
        f'--Xpreproc="-DCONV_NOQDQ_A8W8={1}"',
    ]


def aiesimulator_args() -> List[str]:
    return [
        'aiesimulator',
        '--dump-vcd=trace',
        '--profile',
        f'--pkg-dir={WORKDIR}',
    ]

shape_table = [
    (
        26, 26, 64,  # Yo, Xo, Co
        13, 13, 64,  # Yi, Xi, Ci
        0,          # mode: 0->half_pixel, 1->align_corners, 2->asymmetric
    ),
]


def main():
    assert len(sys.argv) in (1, 3)
    run_mode = sys.argv[1]
    # check if mode is "dataflow" or "sim" or "cert"
    if run_mode not in ('dataflow', 'sim', 'cert'):
        print(f"Invalid mode: {run_mode}. Use 'dataflow', 'sim', or 'cert'.")
        sys.exit(1)
    shape_index = int(sys.argv[2]) if len(sys.argv) == 3 else None
    if shape_index is not None:
        to_run_shape_table = [shape_table[shape_index]] 
    for (
        Yo, Xo, Co,
        Yi, Xi, Ci,
        mode,
    ) in to_run_shape_table:
        Coordinatemode = CoordinateTransfromationMode.HALF_PIXEL
        if mode == 1:
            Coordinatemode = CoordinateTransfromationMode.ALIGN_CORNERS
        elif mode == 2:
            Coordinatemode = CoordinateTransfromationMode.ASYMMETRIC
        input_subv_dims = resize_nni_input_subvol_dims(
            Coordinatemode,
            (Yo, Xo),
            (Yi, Xi),
            (Yo, Xo),
            (1, 1),
            True,
        )
        (
            (Yis, Xis),
            (Yis_step, Xis_step),
            (Yis_offset, Xis_offset),
        ) = input_subv_dims
        print(F"mode: {Coordinatemode}")

        dims = ResizeNNIDims(
            coord_mode=mode, N=1,
            Yo=Yo, Xo=Xo, Co=Co,
            Yi=Yi, Xi=Xi, Ci=Ci,
            Yos=Yo, Xos=Xo, Cos=Co,
            Yis=Yis, Xis=Xis, Cis=Ci,
            N_split=1, Co_split=1, Y_split=1, X_split=1,
            N_loop=1, Co_loop=1, Y_loop=1, X_loop=1, Ci_loop=1,
            aie_cols=1, aie_rows=1,
            Yis_size=Yis, Xis_size=Xis,
            Yis_step=Yis_step, Xis_step=Xis_step,
            Yis_offset=Yis_offset, Xis_offset=Xis_offset,
            ifm_bits=8, ofm_bits=8,
        )
        compile_resize_nni_dataflow(dims, BackEnd.CertAsm if run_mode == 'cert' else BackEnd.Adf)
        compile_args = aiecompiler_args(dims, run_mode)
        if run_mode != 'dataflow':
            if os.path.isdir(WORKDIR):
                print("Deleting existing Work folder")
                subprocess.run(
                    "rm -r Work",
                    shell=True
                )
            os.makedirs(WORKDIR)
            os.chdir(WORKDIR)
            subprocess.run(' '.join(compile_args), shell=True)
            subprocess.run(
                f"sed -i 's/-ladf_api/-ladf_rt_ctrl_api -ladf_api/g' {WORKDIR}/ps/c_rts/systemC/Makefile",
                shell=True
            )
            subprocess.run(
                f'make -C {WORKDIR}/ps/c_rts/systemC/ all',
                shell=True
            )
            aiesim_args = aiesimulator_args()
            subprocess.run(' '.join(aiesim_args), shell=True)


if __name__ == '__main__':
    main() 
