"""Types for buffer allocation"""

from dataclasses import dataclass
from typing import Optional, NamedTuple
from tiler.base_tiler import HW_CONFIG


@dataclass
class BufferSpec:
    """Specification for a buffer to allocate"""

    name: str
    size: int
    is_ping: bool = False
    is_pong: bool = False
    priority: int = 0  # Higher priority = must allocate
    alignment: int = HW_CONFIG.MEMORY_ALIGNMENT  # Alignment in bytes

    def __post_init__(self):
        if self.size < 0:
            raise ValueError("Buffer size must be positive")
        if self.is_ping and self.is_pong:
            raise ValueError("Buffer cannot be both ping and pong")
        if not self.priority >= 0:
            raise ValueError("Priority must be non-negative")
        if self.alignment and self.alignment & (self.alignment - 1) != 0:
            raise ValueError("Alignment must be a power of 2")

    def __hash__(self):
        return hash((self.name, self.size, self.is_ping, self.is_pong, self.priority, self.alignment))


@dataclass
class BufferAllocation:
    """Result of buffer allocation"""

    name: str
    size: int
    addr: Optional[int]


@dataclass
class BinItem:
    """Item to be packed into bins"""

    name: str
    size: int
    alignment: int = 1  # Must be power of 2
    priority: int = 1  # Higher = more important
    must_place: bool = False  # Must be in solution

    def __post_init__(self):
        # Validate alignment is power of 2
        if self.alignment & (self.alignment - 1) != 0:
            raise ValueError(f"Alignment {self.alignment} must be power of 2")


@dataclass
class AllocationResult:
    """Result of buffer allocation with convenient query methods"""

    allocations: list[BufferAllocation]

    def get(self, name: str) -> Optional[BufferAllocation]:
        """Get allocation by name, returns None if not found"""
        for alloc in self.allocations:
            if alloc.name == name:
                return alloc
        return None

    def get_addr(self, name: str) -> Optional[int]:
        """Get address by name, returns None if not found or not allocated"""
        alloc = self.get(name)
        return alloc.addr if alloc else None

    def get_size(self, name: str) -> Optional[int]:
        """Get size by name, returns None if not found"""
        alloc = self.get(name)
        return alloc.size if alloc else None

    def has(self, name: str) -> bool:
        """Check if allocation exists"""
        return self.get(name) is not None

    def is_allocated(self, name: str) -> bool:
        """Check if buffer was successfully allocated (has address)"""
        alloc = self.get(name)
        return alloc is not None and alloc.addr is not None

    def __getitem__(self, name: str) -> BufferAllocation:
        """Get allocation by name using dict-like syntax, raises KeyError if not found"""
        alloc = self.get(name)
        if alloc is None:
            raise KeyError(f"Buffer '{name}' not found in allocations")
        return alloc

    def __contains__(self, name: str) -> bool:
        """Support 'in' operator"""
        return self.has(name)

    def __len__(self) -> int:
        """Number of allocations"""
        return len(self.allocations)

    def __iter__(self):
        """Iterate over allocations"""
        return iter(self.allocations)


class BufferPair(NamedTuple):
    """Pair of buffers that are mutually exclusive"""
    buffer1: BufferSpec
    buffer2: BufferSpec
